/*------------------------------------------------------------------------------
--       Copyright (c) 2015-2017, VeriSilicon Inc. All rights reserved        --
--         Copyright (c) 2011-2014, Google Inc. All rights reserved.          --
--         Copyright (c) 2007-2010, Hantro OY. All rights reserved.           --
--                                                                            --
-- This software is confidential and proprietary and may be used only as      --
--   expressly authorized by VeriSilicon in a written licensing agreement.    --
--                                                                            --
--         This entire notice must be reproduced on all copies                --
--                       and may not be removed.                              --
--                                                                            --
--------------------------------------------------------------------------------
-- Redistribution and use in source and binary forms, with or without         --
-- modification, are permitted provided that the following conditions are met:--
--   * Redistributions of source code must retain the above copyright notice, --
--       this list of conditions and the following disclaimer.                --
--   * Redistributions in binary form must reproduce the above copyright      --
--       notice, this list of conditions and the following disclaimer in the  --
--       documentation and/or other materials provided with the distribution. --
--   * Neither the names of Google nor the names of its contributors may be   --
--       used to endorse or promote products derived from this software       --
--       without specific prior written permission.                           --
--------------------------------------------------------------------------------
-- THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"--
-- AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE  --
-- IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE --
-- ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE  --
-- LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR        --
-- CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF       --
-- SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS   --
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN    --
-- CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)    --
-- ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE --
-- POSSIBILITY OF SUCH DAMAGE.                                                --
--------------------------------------------------------------------------------
--                                                                            --
--  Abstract  :   Entropy tables
--
------------------------------------------------------------------------------*/

#ifndef VP8ENTROPY_TABLE_H
#define VP8ENTROPY_TABLE_H

/*------------------------------------------------------------------------------
    1. Include headers
------------------------------------------------------------------------------*/
#include "vp8putbits.h"

/*------------------------------------------------------------------------------
    2. External compiler flags
--------------------------------------------------------------------------------

--------------------------------------------------------------------------------
    3. Module defines
------------------------------------------------------------------------------*/

/* Luma trees of keyframe and not key frame and default propability */
static const tree const Ymode[] = {
	{0, 1, {0}},			/* (kf) B_PRED  / DC_PRED   0 */
	{4, 3, {0,1,2}},		/* (kf) DC_PRED / V_PRED  100 */
	{5, 3, {0,1,2}},		/* (kf) V_PRED  / H_PRED  101 */
	{6, 3, {0,1,3}},		/* (kf) H_PRED  / TM_PRED 110 */
	{7, 3, {0,1,3}}			/* (kf) TM_PRED / B_PRED  111 */
};

static const tree const *YmodeTree[] = {
	Ymode,				/* prediction type = DC_PRED */
	Ymode + 1,			/* prediction type = V_PRED  */
	Ymode + 2,			/* prediction type = H_PRED  */
	Ymode + 3,			/* prediction type = TM_PRED */
	Ymode + 4			/* prediction type = B_PRED  */
};

static const i32 const kfYmodeProb[4] = {
	145, 156, 163, 128
};

static const i32 const YmodeProb[4] = {
	112, 86, 140, 37
};

/* Luma tree of subblocks */
static const tree const Bmode[] = {
	{0,   1, {0}},			/* B_DC_PRED,        0 */
	{2,   2, {0,1}},		/* B_TM_PRED,       10 */
	{6,   3, {0,1,2}},		/* B_VE_PRED,      110 */
	{28,  5, {0,1,2,3,4}},		/* B_HE_PRED,    11100 */
	{58,  6, {0,1,2,3,4,5}},	/* B_LD_PRED,   111010 */
	{59,  6, {0,1,2,3,4,5}},	/* B_RD_PRED,   111011 */
	{30,  5, {0,1,2,3,6}},		/* B_VR_PRED,    11110 */
	{62,  6, {0,1,2,3,6,7}},	/* B_VL_PRED,   111110 */
	{126, 7, {0,1,2,3,6,7,8}},	/* B_HD_PRED,  1111110 */
	{127, 7, {0,1,2,3,6,7,8}}	/* B_HU_PRED,  1111111 */
};

static const tree const *BmodeTree[] = {
	NULL,				/* prediction type = DC_PRED,     */
	NULL,				/* prediction type = V_PRED,      */
	NULL,				/* prediction type = H_PRED,      */
	NULL,				/* prediction type = TM_PRED,     */
	NULL,				/* prediction type = B_PRED,      */
	Bmode,				/* prediction type = B_DC_PRED,   */
	Bmode + 1,			/* prediction type = B_TM_PRED,   */
	Bmode + 2,			/* prediction type = B_VE_PRED,   */
	Bmode + 3,			/* prediction type = B_HE_PRED,   */
	Bmode + 6,			/* prediction type = B_LD_PRED,   */
	Bmode + 4,			/* prediction type = B_RD_PRED,   */
	Bmode + 5,			/* prediction type = B_VR_PRED,   */
	Bmode + 7,			/* prediction type = B_VL_PRED,   */
	Bmode + 8,			/* prediction type = B_HD_PRED,   */
	Bmode + 9			/* prediction type = B_HU_PRED,   */
};

static const i32 const kfBmodeProb[10][10][9] = {
	{
		{231, 120,  48,  89, 115, 113, 120, 152, 112},
		{152, 179,  64, 126, 170, 118,  46,  70,  95},
		{175,  69, 143,  80,  85,  82,  72, 155, 103},
		{ 56,  58,  10, 171, 218, 189,  17,  13, 152},
		{144,  71,  10,  38, 171, 213, 144,  34,  26},
		{114,  26,  17, 163,  44, 195,  21,  10, 173},
		{121,  24,  80, 195,  26,  62,  44,  64,  85},
		{170,  46,  55,  19, 136, 160,  33, 206,  71},
		{ 63,  20,   8, 114, 114, 208,  12,   9, 226},
		{ 81,  40,  11,  96, 182,  84,  29,  16,  36}
	},

	{
		{134, 183,  89, 137,  98, 101, 106, 165, 148},
		{ 72, 187, 100, 130, 157, 111,  32,  75,  80},
		{ 66, 102, 167,  99,  74,  62,  40, 234, 128},
		{ 41,  53,   9, 178, 241, 141,  26,   8, 107},
		{104,  79,  12,  27, 217, 255,  87,  17,   7},
		{ 74,  43,  26, 146,  73, 166,  49,  23, 157},
		{ 65,  38, 105, 160,  51,  52,  31, 115, 128},
		{ 87,  68,  71,  44, 114,  51,  15, 186,  23},
		{ 47,  41,  14, 110, 182, 183,  21,  17, 194},
		{ 66,  45,  25, 102, 197, 189,  23,  18,  22}
	},

	{
		{ 88,  88, 147, 150,  42,  46,  45, 196, 205},
		{ 43,  97, 183, 117,  85,  38,  35, 179,  61},
		{ 39,  53, 200,  87,  26,  21,  43, 232, 171},
		{ 56,  34,  51, 104, 114, 102,  29,  93,  77},
		{107,  54,  32,  26,  51,   1,  81,  43,  31},
		{ 39,  28,  85, 171,  58, 165,  90,  98,  64},
		{ 34,  22, 116, 206,  23,  34,  43, 166,  73},
		{ 68,  25, 106,  22,  64, 171,  36, 225, 114},
		{ 34,  19,  21, 102, 132, 188,  16,  76, 124},
		{ 62,  18,  78,  95,  85,  57,  50,  48,  51}
	},

	{
		{193, 101,  35, 159, 215, 111,  89,  46, 111},
		{ 60, 148,  31, 172, 219, 228,  21,  18, 111},
		{112, 113,  77,  85, 179, 255,  38, 120, 114},
		{ 40,  42,   1, 196, 245, 209,  10,  25, 109},
		{100,  80,   8,  43, 154,   1,  51,  26,  71},
		{ 88,  43,  29, 140, 166, 213,  37,  43, 154},
		{ 61,  63,  30, 155,  67,  45,  68,   1, 209},
		{142,  78,  78,  16, 255, 128,  34, 197, 171},
		{ 41,  40,   5, 102, 211, 183,   4,   1, 221},
		{ 51,  50,  17, 168, 209, 192,  23,  25,  82}
	},

	{
		{125,  98,  42,  88, 104,  85, 117, 175,  82},
		{ 95,  84,  53,  89, 128, 100, 113, 101,  45},
		{ 75,  79, 123,  47,  51, 128,  81, 171,   1},
		{ 57,  17,   5,  71, 102,  57,  53,  41,  49},
		{115,  21,   2,  10, 102, 255, 166,  23,   6},
		{ 38,  33,  13, 121,  57,  73,  26,   1,  85},
		{ 41,  10,  67, 138,  77, 110,  90,  47, 114},
		{101,  29,  16,  10,  85, 128, 101, 196,  26},
		{ 57,  18,  10, 102, 102, 213,  34,  20,  43},
		{117,  20,  15,  36, 163, 128,  68,   1,  26}
	},

	{
		{138,  31,  36, 171,  27, 166,  38,  44, 229},
		{ 67,  87,  58, 169,  82, 115,  26,  59, 179},
		{ 63,  59,  90, 180,  59, 166,  93,  73, 154},
		{ 40,  40,  21, 116, 143, 209,  34,  39, 175},
		{ 57,  46,  22,  24, 128,   1,  54,  17,  37},
		{ 47,  15,  16, 183,  34, 223,  49,  45, 183},
		{ 46,  17,  33, 183,   6,  98,  15,  32, 183},
		{ 65,  32,  73, 115,  28, 128,  23, 128, 205},
		{ 40,   3,   9, 115,  51, 192,  18,   6, 223},
		{ 87,  37,   9, 115,  59,  77,  64,  21,  47}
	},

	{
		{104,  55,  44, 218,   9,  54,  53, 130, 226},
		{ 64,  90,  70, 205,  40,  41,  23,  26,  57},
		{ 54,  57, 112, 184,   5,  41,  38, 166, 213},
		{ 30,  34,  26, 133, 152, 116,  10,  32, 134},
		{ 75,  32,  12,  51, 192, 255, 160,  43,  51},
		{ 39,  19,  53, 221,  26, 114,  32,  73, 255},
		{ 31,   9,  65, 234,   2,  15,   1, 118,  73},
		{ 88,  31,  35,  67, 102,  85,  55, 186,  85},
		{ 56,  21,  23, 111,  59, 205,  45,  37, 192},
		{ 55,  38,  70, 124,  73, 102,   1,  34,  98}
	},

	{
		{102,  61,  71,  37,  34,  53,  31, 243, 192},
		{ 69,  60,  71,  38,  73, 119,  28, 222,  37},
		{ 68,  45, 128,  34,   1,  47,  11, 245, 171},
		{ 62,  17,  19,  70, 146,  85,  55,  62,  70},
		{ 75,  15,   9,   9,  64, 255, 184, 119,  16},
		{ 37,  43,  37, 154, 100, 163,  85, 160,   1},
		{ 63,   9,  92, 136,  28,  64,  32, 201,  85},
		{ 86,   6,  28,   5,  64, 255,  25, 248,   1},
		{ 56,   8,  17, 132, 137, 255,  55, 116, 128},
		{ 58,  15,  20,  82, 135,  57,  26, 121,  40}
	},

	{
		{164,  50,  31, 137, 154, 133,  25,  35, 218},
		{ 51, 103,  44, 131, 131, 123,  31,   6, 158},
		{ 86,  40,  64, 135, 148, 224,  45, 183, 128},
		{ 22,  26,  17, 131, 240, 154,  14,   1, 209},
		{ 83,  12,  13,  54, 192, 255,  68,  47,  28},
		{ 45,  16,  21,  91,  64, 222,   7,   1, 197},
		{ 56,  21,  39, 155,  60, 138,  23, 102, 213},
		{ 85,  26,  85,  85, 128, 128,  32, 146, 171},
		{ 18,  11,   7,  63, 144, 171,   4,   4, 246},
		{ 35,  27,  10, 146, 174, 171,  12,  26, 128}
	},

	{
		{190,  80,  35,  99, 180,  80, 126,  54,  45},
		{ 85, 126,  47,  87, 176,  51,  41,  20,  32},
		{101,  75, 128, 139, 118, 146, 116, 128,  85},
		{ 56,  41,  15, 176, 236,  85,  37,   9,  62},
		{146,  36,  19,  30, 171, 255,  97,  27,  20},
		{ 71,  30,  17, 119, 118, 255,  17,  18, 138},
		{101,  38,  60, 138,  55,  70,  43,  26, 142},
		{138,  45,  61,  62, 219,   1,  81, 188,  64},
		{ 32,  41,  20, 117, 151, 142,  20,  21, 163},
		{112,  19,  12,  61, 195, 128,  48,   4,  24}
	}
};

static const i32 const BmodeProb[9] = {
	120, 90, 79, 133, 87, 85, 80, 111, 151
};

static const i32 const kfUVmodeProb[3] = {
	142, 114, 183
};

static const i32 const UVmodeProb[3] = {
	162, 101, 204
};

/* DCT */
static const tree const dctTree[] = {
	{2,   2, {0,1}},		/* DCT_0      10 */
	{6,   3, {0,1,2}},		/* DCT_1     110 */
	{28,  5, {0,1,2,3,4}},		/* DCT_2   11100 */
	{58,  6, {0,1,2,3,4,5}},	/* DCT_3  111010 */
	{59,  6, {0,1,2,3,4,5}},	/* DCT_4  111011 */
	{60,  6, {0,1,2,3,6,7}},	/* CAT_1  111100 */
	{61,  6, {0,1,2,3,6,7}},	/* CAT_2  111101 */
	{124, 7, {0,1,2,3,6,8,9}},	/* CAT_3 1111100 */
	{125, 7, {0,1,2,3,6,8,9}},	/* CAT_4 1111101 */
	{126, 7, {0,1,2,3,6,8,10}},	/* CAT_5 1111110 */
	{127, 7, {0,1,2,3,6,8,10}},	/* CAT_6 1111111 */
	{0,   1, {0}}			/* EOB         0 */
};

/* No EOB follows a DCT_0, so we can skip the first branch of
 * the above table. This table is used if prev coeff = zero */
static const tree const dctTreeNoEOB[] = {
	{0,   1, {1}},			/* DCT_0      0 */
	{2,   2, {1,2}},		/* DCT_1     10 */
	{12,  4, {1,2,3,4}},		/* DCT_2   1100 */
	{26,  5, {1,2,3,4,5}},		/* DCT_3  11010 */
	{27,  5, {1,2,3,4,5}},		/* DCT_4  11011 */
	{28,  5, {1,2,3,6,7}},		/* CAT_1  11100 */
	{29,  5, {1,2,3,6,7}},		/* CAT_2  11101 */
	{60,  6, {1,2,3,6,8,9}},	/* CAT_3 111100 */
	{61,  6, {1,2,3,6,8,9}},	/* CAT_4 111101 */
	{62,  6, {1,2,3,6,8,10}},	/* CAT_5 111110 */
	{63,  6, {1,2,3,6,8,10}},	/* CAT_6 111111 */
};

static const i32 const defaultCoeffProb[4][8][3][11] = {
	{
		{
			{128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128},
			{128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128},
			{128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128}
		},
		{
			{253, 136, 254, 255, 228, 219, 128, 128, 128, 128, 128},
			{189, 129, 242, 255, 227, 213, 255, 219, 128, 128, 128},
			{106, 126, 227, 252, 214, 209, 255, 255, 128, 128, 128}
		},
		{
			{  1,  98, 248, 255, 236, 226, 255, 255, 128, 128, 128},
			{181, 133, 238, 254, 221, 234, 255, 154, 128, 128, 128},
			{ 78, 134, 202, 247, 198, 180, 255, 219, 128, 128, 128}
		},
		{
			{  1, 185, 249, 255, 243, 255, 128, 128, 128, 128, 128},
			{184, 150, 247, 255, 236, 224, 128, 128, 128, 128, 128},
			{ 77, 110, 216, 255, 236, 230, 128, 128, 128, 128, 128}
		},
		{
			{  1, 101, 251, 255, 241, 255, 128, 128, 128, 128, 128},
			{170, 139, 241, 252, 236, 209, 255, 255, 128, 128, 128},
			{ 37, 116, 196, 243, 228, 255, 255, 255, 128, 128, 128}
		},
		{
			{  1, 204, 254, 255, 245, 255, 128, 128, 128, 128, 128},
			{207, 160, 250, 255, 238, 128, 128, 128, 128, 128, 128},
			{102, 103, 231, 255, 211, 171, 128, 128, 128, 128, 128}
		},
		{
			{  1, 152, 252, 255, 240, 255, 128, 128, 128, 128, 128},
			{177, 135, 243, 255, 234, 225, 128, 128, 128, 128, 128},
			{ 80, 129, 211, 255, 194, 224, 128, 128, 128, 128, 128}
		},
		{
			{  1,   1, 255, 128, 128, 128, 128, 128, 128, 128, 128},
			{246,   1, 255, 128, 128, 128, 128, 128, 128, 128, 128},
			{255, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128}
		}
	},

	{
		{
			{198,  35, 237, 223, 193, 187, 162, 160, 145, 155,  62},
			{131,  45, 198, 221, 172, 176, 220, 157, 252, 221,   1},
			{ 68,  47, 146, 208, 149, 167, 221, 162, 255, 223, 128}
		},
		{
			{  1, 149, 241, 255, 221, 224, 255, 255, 128, 128, 128},
			{184, 141, 234, 253, 222, 220, 255, 199, 128, 128, 128},
			{ 81,  99, 181, 242, 176, 190, 249, 202, 255, 255, 128}
		},
		{
			{  1, 129, 232, 253, 214, 197, 242, 196, 255, 255, 128},
			{ 99, 121, 210, 250, 201, 198, 255, 202, 128, 128, 128},
			{ 23,  91, 163, 242, 170, 187, 247, 210, 255, 255, 128}
		},
		{
			{  1, 200, 246, 255, 234, 255, 128, 128, 128, 128, 128},
			{109, 178, 241, 255, 231, 245, 255, 255, 128, 128, 128},
			{ 44, 130, 201, 253, 205, 192, 255, 255, 128, 128, 128}
		},
		{
			{  1, 132, 239, 251, 219, 209, 255, 165, 128, 128, 128},
			{ 94, 136, 225, 251, 218, 190, 255, 255, 128, 128, 128},
			{ 22, 100, 174, 245, 186, 161, 255, 199, 128, 128, 128}
		},
		{
			{  1, 182, 249, 255, 232, 235, 128, 128, 128, 128, 128},
			{124, 143, 241, 255, 227, 234, 128, 128, 128, 128, 128},
			{ 35,  77, 181, 251, 193, 211, 255, 205, 128, 128, 128}
		},
		{
			{  1, 157, 247, 255, 236, 231, 255, 255, 128, 128, 128},
			{121, 141, 235, 255, 225, 227, 255, 255, 128, 128, 128},
			{ 45,  99, 188, 251, 195, 217, 255, 224, 128, 128, 128}
		},
		{
			{  1,   1, 251, 255, 213, 255, 128, 128, 128, 128, 128},
			{203,   1, 248, 255, 255, 128, 128, 128, 128, 128, 128},
			{137,   1, 177, 255, 224, 255, 128, 128, 128, 128, 128}
		}
	},

	{
		{
			{253,   9, 248, 251, 207, 208, 255, 192, 128, 128, 128},
			{175,  13, 224, 243, 193, 185, 249, 198, 255, 255, 128},
			{ 73,  17, 171, 221, 161, 179, 236, 167, 255, 234, 128}
		},
		{
			{  1,  95, 247, 253, 212, 183, 255, 255, 128, 128, 128},
			{239,  90, 244, 250, 211, 209, 255, 255, 128, 128, 128},
			{155,  77, 195, 248, 188, 195, 255, 255, 128, 128, 128}
		},
		{
			{  1,  24, 239, 251, 218, 219, 255, 205, 128, 128, 128},
			{201,  51, 219, 255, 196, 186, 128, 128, 128, 128, 128},
			{ 69,  46, 190, 239, 201, 218, 255, 228, 128, 128, 128}
		},
		{
			{  1, 191, 251, 255, 255, 128, 128, 128, 128, 128, 128},
			{223, 165, 249, 255, 213, 255, 128, 128, 128, 128, 128},
			{141, 124, 248, 255, 255, 128, 128, 128, 128, 128, 128}
		},
		{
			{  1,  16, 248, 255, 255, 128, 128, 128, 128, 128, 128},
			{190,  36, 230, 255, 236, 255, 128, 128, 128, 128, 128},
			{149,   1, 255, 128, 128, 128, 128, 128, 128, 128, 128}
		},
		{
			{  1, 226, 255, 128, 128, 128, 128, 128, 128, 128, 128},
			{247, 192, 255, 128, 128, 128, 128, 128, 128, 128, 128},
			{240, 128, 255, 128, 128, 128, 128, 128, 128, 128, 128}
		},
		{
			{  1, 134, 252, 255, 255, 128, 128, 128, 128, 128, 128},
			{213,  62, 250, 255, 255, 128, 128, 128, 128, 128, 128},
			{ 55,  93, 255, 128, 128, 128, 128, 128, 128, 128, 128}
		},
		{
			{128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128},
			{128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128},
			{128, 128, 128, 128, 128, 128, 128, 128, 128, 128, 128}
		}
	},

	{
		{
			{202,  24, 213, 235, 186, 191, 220, 160, 240, 175, 255},
			{126,  38, 182, 232, 169, 184, 228, 174, 255, 187, 128},
			{ 61,  46, 138, 219, 151, 178, 240, 170, 255, 216, 128}
		},
		{
			{  1, 112, 230, 250, 199, 191, 247, 159, 255, 255, 128},
			{166, 109, 228, 252, 211, 215, 255, 174, 128, 128, 128},
			{ 39,  77, 162, 232, 172, 180, 245, 178, 255, 255, 128}
		},
		{
			{  1,  52, 220, 246, 198, 199, 249, 220, 255, 255, 128},
			{124,  74, 191, 243, 183, 193, 250, 221, 255, 255, 128},
			{ 24,  71, 130, 219, 154, 170, 243, 182, 255, 255, 128}
		},
		{
			{  1, 182, 225, 249, 219, 240, 255, 224, 128, 128, 128},
			{149, 150, 226, 252, 216, 205, 255, 171, 128, 128, 128},
			{ 28, 108, 170, 242, 183, 194, 254, 223, 255, 255, 128}
		},
		{
			{  1,  81, 230, 252, 204, 203, 255, 192, 128, 128, 128},
			{123, 102, 209, 247, 188, 196, 255, 233, 128, 128, 128},
			{ 20,  95, 153, 243, 164, 173, 255, 203, 128, 128, 128}
		},
		{
			{  1, 222, 248, 255, 216, 213, 128, 128, 128, 128, 128},
			{168, 175, 246, 252, 235, 205, 255, 255, 128, 128, 128},
			{ 47, 116, 215, 255, 211, 212, 255, 255, 128, 128, 128}
		},
		{
			{  1, 121, 236, 253, 212, 214, 255, 255, 128, 128, 128},
			{141,  84, 213, 252, 201, 202, 255, 219, 128, 128, 128},
			{ 42,  80, 160, 240, 162, 185, 255, 205, 128, 128, 128}
		},
		{
			{  1,   1, 255, 128, 128, 128, 128, 128, 128, 128, 128},
			{244,   1, 255, 128, 128, 128, 128, 128, 128, 128, 128},
			{238,   1, 255, 128, 128, 128, 128, 128, 128, 128, 128}
		}
	}
};

static const i32 const coeffUpdateProb[4][8][3][11] = {
	{
		{
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{176, 246, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{223, 241, 252, 255, 255, 255, 255, 255, 255, 255, 255},
			{249, 253, 253, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 244, 252, 255, 255, 255, 255, 255, 255, 255, 255},
			{234, 254, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{253, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 246, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{239, 253, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{254, 255, 254, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 248, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{251, 255, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 253, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{251, 254, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{254, 255, 254, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 254, 253, 255, 254, 255, 255, 255, 255, 255, 255},
			{250, 255, 254, 255, 254, 255, 255, 255, 255, 255, 255},
			{254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		}
	},
	{
		{
			{217, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{225, 252, 241, 253, 255, 255, 254, 255, 255, 255, 255},
			{234, 250, 241, 250, 253, 255, 253, 254, 255, 255, 255}
		},
		{
			{255, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{223, 254, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{238, 253, 254, 254, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 248, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{249, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 253, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{247, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 253, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{252, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 254, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{253, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 254, 253, 255, 255, 255, 255, 255, 255, 255, 255},
			{250, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		}
	},

	{
		{
			{186, 251, 250, 255, 255, 255, 255, 255, 255, 255, 255},
			{234, 251, 244, 254, 255, 255, 255, 255, 255, 255, 255},
			{251, 251, 243, 253, 254, 255, 254, 255, 255, 255, 255}
		},
		{
			{255, 253, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{236, 253, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{251, 253, 253, 254, 254, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 254, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{254, 254, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{254, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		}
	},

	{
		{
			{248, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{250, 254, 252, 254, 255, 255, 255, 255, 255, 255, 255},
			{248, 254, 249, 253, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 253, 253, 255, 255, 255, 255, 255, 255, 255, 255},
			{246, 253, 253, 255, 255, 255, 255, 255, 255, 255, 255},
			{252, 254, 251, 254, 254, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 254, 252, 255, 255, 255, 255, 255, 255, 255, 255},
			{248, 254, 253, 255, 255, 255, 255, 255, 255, 255, 255},
			{253, 255, 254, 254, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 251, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{245, 251, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{253, 253, 254, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 251, 253, 255, 255, 255, 255, 255, 255, 255, 255},
			{252, 253, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 254, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 252, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{249, 255, 254, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 254, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 255, 253, 255, 255, 255, 255, 255, 255, 255, 255},
			{250, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		},
		{
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{254, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255},
			{255, 255, 255, 255, 255, 255, 255, 255, 255, 255, 255}
		}
	}
};

static const i32 const mvUpdateProb[2][19] = {
	{237, 246, 253, 253, 254, 254, 254, 254, 254, 254,
	 254, 254, 254, 254, 250, 250, 252, 254, 254},

	{231, 243, 245, 253, 254, 254, 254, 254, 254, 254,
	 254, 254, 254, 254, 251, 251, 254, 254, 254}
};

static const i32 const defaultMvProb[2][19] = {
	{162, 128, 225, 146, 172, 147, 214,  39, 156, 128,
	 129, 132,  75, 145, 178, 206, 239, 254, 254},

	{164, 128, 204, 170, 119, 235, 140, 230, 228, 128,
	 130, 130,  74, 148, 180, 203, 236, 254, 254}
};

static const i32 const subMvPartProb[3] = {
	110, 111, 150
};

static const i32 const subMvRefProb[5][3] = {
	{147, 136, 18},
	{106, 145,  1},
	{179, 121,  1},
	{223,   1, 34},
	{208,   1,  1}
};

/* Motion vector tree */
static const tree const mvTree[] = {
	{0, 3, {0,1,2}},		/* mv_0 000 */
	{1, 3, {0,1,2}},		/* mv_1 001 */
	{2, 3, {0,1,3}},		/* mv_2 010 */
	{3, 3, {0,1,3}},		/* mv_3 011 */
	{4, 3, {0,4,5}},		/* mv_4 100 */
	{5, 3, {0,4,5}},		/* mv_5 101 */
	{6, 3, {0,4,6}},		/* mv_6 110 */
	{7, 3, {0,4,6}},		/* mv_7 111 */
};

/* Segment tree and default propability */
static const tree const segmentTree[] = {
	{0, 2, {0,1}},			/* segmentId = 0, 00 */
	{1, 2, {0,1}},			/* segmentId = 1, 01 */
	{2, 2, {0,2}},			/* segmentId = 2, 10 */
	{3, 2, {0,2}},			/* segmentId = 3, 11 */
};

static const i32 const segmentProb[3] = {
	255, 255, 255
};

/* If probability being zero is p, then avarage bits used when bool is
zero = log2(1/p) and when bool is one = log2(1/(1-p)).

For example bins probability being zero is p = 0.5
bin = 0 -> average bits used is log2(1/0.5)      = 1 bits/bin
bin = 1 -> average bits used is log2(1/(1 - 0.5) = 1 bits/bin

For example bins probability being zero is p = 0.95
bin = 0 -> average bits used is log2(1/0.95)      = 0.074 bits/bin
bin = 1 -> average bits used is log2(1/(1 - 0.95) = 4.321 bits/bin

Table cost[] below is calculated as follow: cost[p] is zero bin's average bit
cost at given p = [1..255] (note that probability is p/256) scaled up by SCALE.
for (i = 0; i < 256; i++) cost[i] = round((log2((double)256/i) * SCALE)).
Magic number SCALE = 256. */
static const i32 const vp8_prob_cost[] = {
	2048, 2048, 1792, 1642, 1536, 1454, 1386, 1329, 1280, 1236,
	1198, 1162, 1130, 1101, 1073, 1048, 1024, 1002,  980,  961,
	 942,  924,  906,  890,  874,  859,  845,  831,  817,  804,
	 792,  780,  768,  757,  746,  735,  724,  714,  705,  695,
	 686,  676,  668,  659,  650,  642,  634,  626,  618,  611,
	 603,  596,  589,  582,  575,  568,  561,  555,  548,  542,
	 536,  530,  524,  518,  512,  506,  501,  495,  490,  484,
	 479,  474,  468,  463,  458,  453,  449,  444,  439,  434,
	 430,  425,  420,  416,  412,  407,  403,  399,  394,  390,
	 386,  382,  378,  374,  370,  366,  362,  358,  355,  351,
	 347,  343,  340,  336,  333,  329,  326,  322,  319,  315,
	 312,  309,  305,  302,  299,  296,  292,  289,  286,  283,
	 280,  277,  274,  271,  268,  265,  262,  259,  256,  253,
	 250,  247,  245,  242,  239,  236,  234,  231,  228,  226,
	 223,  220,  218,  215,  212,  210,  207,  205,  202,  200,
	 197,  195,  193,  190,  188,  185,  183,  181,  178,  176,
	 174,  171,  169,  167,  164,  162,  160,  158,  156,  153,
	 151,  149,  147,  145,  143,  140,  138,  136,  134,  132,
	 130,  128,  126,  124,  122,  120,  118,  116,  114,  112,
	 110,  108,  106,  104,  102,  101,   99,   97,   95,   93,
	  91,   89,   87,   86,   84,   82,   80,   78,   77,   75,
	  73,   71,   70,   68,   66,   64,   63,   61,   59,   58,
	  56,   54,   53,   51,   49,   48,   46,   44,   43,   41,
	  40,   38,   36,   35,   33,   32,   30,   28,   27,   25,
	  24,   22,   21,   19,   18,   16,   15,   13,   12,   10,
	   9,    7,    6,    4,    3,    1
};

static const i32 const defaultSkipFalseProb[128] =
{
    255, 255, 255, 255, 255, 255, 255, 255,
    255, 255, 255, 255, 255, 255, 255, 255,
    255, 255, 255, 255, 255, 255, 255, 255,
    255, 255, 255, 255, 255, 255, 255, 255,
    255, 255, 255, 255, 255, 255, 255, 255,
    255, 255, 255, 255, 255, 255, 255, 255,
    255, 255, 255, 255, 255, 255, 255, 255,
    251, 248, 244, 240, 236, 232, 229, 225,
    221, 217, 213, 208, 204, 199, 194, 190,
    187, 183, 179, 175, 172, 168, 164, 160,
    157, 153, 149, 145, 142, 138, 134, 130,
    127, 124, 120, 117, 114, 110, 107, 104,
    101, 98,  95,  92,  89,  86,  83, 80,
    77,  74,  71,  68,  65,  62,  59, 56,
    53,  50,  47,  44,  41,  38,  35, 32,
    30,  28,  26,  24,  22,  20,  18, 16,
};

#endif
