#!/usr/bin/env bash
##
## SPDX-License-Identifier: LGPL-2.1-only
##
## @file runTest.sh
## @author 
## @date 
## @brief SSAT Test Cases for NNStreamer
##


check_empty() 
{ 
    if [ -z "$1" ]; then 
        echo "Variable is empty" 
        return 0 
    else 
        echo "Variable is not empty" 
        return 1 
    fi 
}


if [[ "$SSATAPILOADED" != "1" ]]; then
    SILENT=0
    INDEPENDENT=1
    search="ssat-api.sh"
    source $search
    printf "${Blue}Independent Mode${NC}"
fi

# This is compatible with SSAT (https://github.com/myungjoo/SSAT)
testInit $1

# NNStreamer and plugins path for test
PATH_TO_PLUGIN="../../build"

if [[ -d $PATH_TO_PLUGIN ]]; then
    ini_path="${PATH_TO_PLUGIN}/ext/nnstreamer/tensor_filter"
    if [[ -d ${ini_path} ]]; then
        check=$(ls ${ini_path} | grep neuronsdk.so)
        if [[ ! $check ]]; then
            echo "Cannot find neuronsdk shared lib"
            report
            exit
        fi
    else
        echo "Cannot find ${ini_path}"
    fi
else
    ini_file="/etc/nnstreamer.ini"
    if [[ -f ${ini_file} ]]; then
        path=$(grep "^filters" ${ini_file})
        key=${path%=*}
        value=${path##*=}

        if [[ $key != "filters" ]]; then
            echo "String Error"
            report
            exit
        fi

        if [[ -d ${value} ]]; then
            check=$(ls ${value} | grep neuronsdk.so)
            if [[ ! $check ]]; then
                echo "Cannot find neuronsdk shared lib"
                report
                exit
            fi
        else
            echo "Cannot file ${value}"
            report
            exit
        fi
    else
        echo "Cannot identify nnstreamer.ini"
        report
        exit
    fi
fi

if [ "$SKIPGEN" == "YES" ]; then
    echo "Test Case Generation Skipped"
    sopath=$2
else
    echo "Test Case Generation Started"
    python3 generateTest.py
    sopath=$1
fi

PATH_TO_MODEL="../test_models/models/mobilenet_v1_1.0_224_quant.tflite"
PATH_TO_DLA="../test_models/models/mobilenet_v1_1.0_224_quant.dla"
PATH_TO_LABEL="../test_models/labels/labels.txt"
PATH_TO_IMAGE="../test_models/data/orange.png"

tensor=$(python3 tflite2dla.py ${PATH_TO_MODEL} ${PATH_TO_DLA} 2>&1 >/dev/null)
echo ${tensor}

#gstTest "--gst-plugin-path=${PATH_TO_PLUGIN} filesrc location=${PATH_TO_IMAGE} ! pngdec ! videoscale ! imagefreeze ! videoconvert ! video/x-raw,format=RGB,framerate=0/1 ! tensor_converter ! tensor_filter framework=neuronsdk model=${PATH_TO_DLA} inputlayout=NHWC ${tensor} ! filesink location=neuronsdk.out.log" image-classification 0 0 $PERFORMANCE
#python3 checkLabel.py neuronsdk.out.log ${PATH_TO_LABEL} orange
#testResult $? image_classification "Golden test comparison" 0 1


# Fail test for no tensor properties
gstTest "--gst-plugin-path=${PATH_TO_PLUGIN} filesrc location=${PATH_TO_IMAGE} ! pngdec ! videoscale ! imagefreeze ! videoconvert ! video/x-raw,format=RGB,framerate=0/1 ! tensor_converter ! tensor_filter framework=neuronsdk model=${PATH_TO_DLA} inputlayout=NHWC ! filesink location=neuronsdk.out.log" no-tensor-properties 0 1 $PERFORMANCE


# Fail test for incorrect tensor properties
wrong_tensor="${tensor},1001:1"
gstTest "--gst-plugin-path=${PATH_TO_PLUGIN} filesrc location=${PATH_TO_IMAGE} ! pngdec ! videoscale ! imagefreeze ! videoconvert ! video/x-raw,format=RGB,framerate=0/1 ! tensor_converter ! tensor_filter framework=neuronsdk model=${PATH_TO_DLA} inputlayout=NHWC ${wrong_tensor} ! filesink location=neuronsdk.out.log" incorrect-tensor-properties 0 1 $PERFORMANCE



PATH_TO_MODEL="../test_models/models/sample_1x4x4x4_two_input_one_output.tflite"
PATH_TO_DLA="../test_models/models/sample_1x4x4x4_two_input_one_output.dla"
# This model is made with below simple python script:
#   import tensorflow as tf
#   from tensorflow import keras
#   from tensorflow.keras.layers import *
#   from tensorflow.keras import *
#
#   x1 = Input(batch_shape=(None, 4, 4, 4,))
#   x2 = Input(batch_shape=(None, 4, 4, 4,))
#   output = Add()([x1, x2])
#   model = Model(inputs=[x1, x2], outputs=[output])
#   model.compile()
#   model.summary()
#
#   converter = tf.lite.TFLiteConverter.from_keras_model(model)
#   tflite_model = converter.convert()
#
#   with open("tflite_model_kidd.tflite", "wb") as f:
#       f.write(tflite_model)

tensor=$(python3 tflite2dla.py ${PATH_TO_MODEL} ${PATH_TO_DLA} 2>&1 >/dev/null)
echo ${tensor}

gstTest "--gst-plugin-path=${PATH_TO_PLUGIN} multifilesrc location=\"test_2_input_1_output.dat\" blocksize=-1 num_buffers=2 ! application/octet-stream ! tensor_converter input-dim=4:4:4:1   input-type=float32 ! tee name=t t. ! queue ! mux.sink_0 t. ! queue ! mux.sink_1  tensor_mux name=mux sync_mode=nosync ! queue ! tensor_filter framework=neuronsdk        model=${PATH_TO_DLA} inputlayout=NHWC ${tensor} ! multifilesink location=neuronsdk.out.log" multi-inputs 0 0 $PERFORMANCE
diff test_2_input_1_output.dat.golden neuronsdk.out.log
testResult $? multi-inputs "Golden test comparison" 0 1




PATH_TO_MODEL="../test_models/models/sample_1x4x4x4_two_input_two_output.tflite"
PATH_TO_DLA="../test_models/models/sample_1x4x4x4_two_input_two_output.dla"
# This model is made with below simple python script:
#   import numpy as np 
#   import tensorflow as tf 
#
#   input_1 = tf.keras.Input(batch_shape=(None, 4, 4, 4,), name='input_1') 
#   input_2 = tf.keras.Input(batch_shape=(None, 4, 4, 4,), name='input_2')
#
#   add_1 = tf.constant(10.0, shape=[1, 4, 4, 4])
#   add_2 = tf.constant(20.0, shape=[1, 4, 4, 4])
#
#   output_1 = tf.keras.layers.Add()([input_1, add_1])
#   output_2 = tf.keras.layers.Subtract()([input_2, add_2])
#
#   model_outputs = [output_1, output_2] 
#
#   model = tf.keras.Model(inputs=[input_1, input_2], outputs=model_outputs)
#   converter = tf.lite.TFLiteConverter.from_keras_model(model) 
#   tflite_model = converter.convert() 
#
#   with open('my_model.2x2.tflite', 'wb') as f: 
#	    f.write(tflite_model)

tensor=$(python3 tflite2dla.py ${PATH_TO_MODEL} ${PATH_TO_DLA} 2>&1 >/dev/null)
echo ${tensor}

gstTest "--gst-plugin-path=${PATH_TO_PLUGIN} multifilesrc location=\"test_2_input_2_output.dat\" blocksize=-1 num_buffers=2 ! application/octet-stream ! tensor_converter input-dim=4:4:4:1   input-type=float32 ! tee name=t t. ! queue ! mux.sink_0 t. ! queue ! mux.sink_1  tensor_mux name=mux sync_mode=nosync ! queue ! tensor_filter framework=neuronsdk        model=${PATH_TO_DLA} inputlayout=NHWC ${tensor} ! multifilesink location=neuronsdk.out.log" multi-outputs 0 0 $PERFORMANCE
diff test_2_input_2_output.dat.golden neuronsdk.out.log
testResult $? multi-outputs "Golden test comparison" 0 1


# Cleanup
rm -f info *.log *.dat *.golden

report
