/**
 * GStreamer Tensor_Filter, OpenVino (DLDT) Module
 * Copyright (C) 2019 Wook Song <wook16.song@samsung.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation;
 * version 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 */
/**
 * @file    tensor_filter_openvino.cc
 * @date    23 Dec 2019
 * @brief   Tensor_filter subplugin for OpenVino (DLDT).
 * @see     http://github.com/nnstreamer/nnstreamer
 * @author  Wook Song <wook16.song@samsung.com>
 * @bug     No known bugs except for NYI items
 *
 * This is the per-NN-framework plugin (OpenVino) for tensor_filter.
 */

#include <glib.h>
#include <nnstreamer_log.h>
#define NO_ANONYMOUS_NESTED_STRUCT
#include <nnstreamer_plugin_api_filter.h>
#undef NO_ANONYMOUS_NESTED_STRUCT
#include <tensor_common.h>
#ifdef __OPENVINO_CPU_EXT__
#include <ext_list.hpp>
#endif /* __OPENVINO_CPU_EXT__ */
#include <inference_engine.hpp>
#include <iostream>
#include <string>
#include <vector>
#include <nnstreamer_util.h>
#include "tensor_filter_openvino.hh"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */
void init_filter_openvino (void) __attribute__((constructor));
void fini_filter_openvino (void) __attribute__((destructor));
#ifdef __cplusplus
}
#endif /* __cplusplus */

static const gchar *openvino_accl_support[]
    = { ACCL_NPU_MOVIDIUS_STR, /** ACCL for default and auto config */
        ACCL_NPU_STR, ACCL_CPU_STR, NULL };

std::map<accl_hw, std::string> TensorFilterOpenvino::_nnsAcclHwToOVDevMap = {
  { ACCL_CPU, "CPU" }, { ACCL_NPU, "MYRIAD" }, { ACCL_NPU_MOVIDIUS, "MYRIAD" },
};

const std::string TensorFilterOpenvino::extBin = ".bin";
const std::string TensorFilterOpenvino::extXml = ".xml";

/**
 * @brief Convert the string representing the tensor data type to _nns_tensor_type
 * @param type a std::string representing the tensor data type in InferenceEngine
 * @return _nns_tensor_type corresponding to the tensor data type in InferenceEngine if OK, otherwise _NNS_END
 */
tensor_type
TensorFilterOpenvino::convertFromIETypeStr (std::string type)
{
  if (type[0] == 'U') {
    if (type[1] == '8')
      return _NNS_UINT8;
    else
      return _NNS_UINT16;
  } else if (type[0] == 'I') {
    if (type[1] == '1')
      return _NNS_INT16;
    else if (type[1] == '3')
      return _NNS_INT32;
    else
      return _NNS_INT8;
  } else if (type[0] == 'F') {
    if (type[2] == '3')
      return _NNS_FLOAT32;
    else
      return _NNS_END;
  } else {
    return _NNS_END;
  }
}

/**
 * @brief Convert a tensor container in NNS to a tensor container in IE
 * @param tensorDesc the class that defines a Tensor description to be converted from a GstTensorMemory
 * @param gstTensor the container of a tensor in NNS to be coverted to a tensor container in IE
 * @return a pointer to the Blob which is a container of a tensor in IE if OK, otherwise nullptr
 */
InferenceEngine::Blob::Ptr
TensorFilterOpenvino::convertGstTensorMemoryToBlobPtr (const InferenceEngine::TensorDesc tensorDesc,
    const GstTensorMemory *gstTensor, const tensor_type gstType)
{
  switch (gstType) {
  case _NNS_UINT8:
    return InferenceEngine::Blob::Ptr (new InferenceEngine::TBlob<uint8_t> (
        tensorDesc, (uint8_t *)gstTensor->data, gstTensor->size));
  case _NNS_UINT16:
    return InferenceEngine::Blob::Ptr (new InferenceEngine::TBlob<uint16_t> (
        tensorDesc, (uint16_t *)gstTensor->data, gstTensor->size));
  case _NNS_INT8:
    return InferenceEngine::Blob::Ptr (new InferenceEngine::TBlob<int8_t> (
        tensorDesc, (int8_t *)gstTensor->data, gstTensor->size));
  case _NNS_INT16:
    return InferenceEngine::Blob::Ptr (new InferenceEngine::TBlob<int16_t> (
        tensorDesc, (int16_t *)gstTensor->data, gstTensor->size));
  case _NNS_INT32:
    return InferenceEngine::Blob::Ptr (new InferenceEngine::TBlob<int32_t> (
        tensorDesc, (int32_t *)gstTensor->data, gstTensor->size));
  case _NNS_FLOAT32:
    return InferenceEngine::Blob::Ptr (new InferenceEngine::TBlob<float> (
        tensorDesc, (float *)gstTensor->data, gstTensor->size));
  default:
    return nullptr;
  }
}

/**
 * @brief Check the given hw is supported by the fw or not
 * @param devsVector a reference of a vector of the available device names (the return of _ieCore.GetAvailableDevices ().)
 * @param hw a user-given acceleration device of which the data type is accl_hw
 * @return TRUE if supported
 */
inline bool
TensorFilterOpenvino::isAcclDevSupported (std::vector<std::string> &devsVector, accl_hw hw)
{
  std::vector<std::string>::iterator it;

  it = std::find (devsVector.begin (), devsVector.end (), _nnsAcclHwToOVDevMap[hw]);

  if (it == devsVector.end ()) {
    return FALSE;
  }

  return TRUE;
}

/**
 * @brief Get a path where the model file in XML format is located
 * @return a std::string of the path
 */
std::string
TensorFilterOpenvino::getPathModelXml ()
{
  return this->_pathModelXml;
}

/**
 * @brief Get a path where the model file in bin format is located
 * @return a std::string of the path
 */
std::string
TensorFilterOpenvino::getPathModelBin ()
{
  return this->_pathModelBin;
}

/**
 * @brief TensorFilterOpenvino constructor
 * @param pathModelXml the path of the given model in a XML format
 * @param pathModelBin the path of the given model in a Bin format
 * @return  Nothing
 */
TensorFilterOpenvino::TensorFilterOpenvino (std::string pathModelXml, std::string pathModelBin)
{
  this->_pathModelXml = pathModelXml;
  this->_pathModelBin = pathModelBin;
  (this->_networkReaderCNN).ReadNetwork (this->_pathModelXml);
  (this->_networkReaderCNN).ReadWeights (this->_pathModelBin);
  this->_networkCNN = (this->_networkReaderCNN).getNetwork ();
  this->_inputsDataMap = (this->_networkCNN).getInputsInfo ();
  this->_outputsDataMap = (this->_networkCNN).getOutputsInfo ();
  this->_isLoaded = false;
  this->_hw = ACCL_NONE;
}

/**
 * @brief TensorFilterOpenvino Destructor
 * @return  Nothing
 */
TensorFilterOpenvino::~TensorFilterOpenvino ()
{
  ;
}

/**
 * @brief Load the given neural network into the target device
 * @param hw a user-given acceleration device to use
 * @return 0 (TensorFilterOpenvino::RetSuccess) if OK, negative values if error
 */
int
TensorFilterOpenvino::loadModel (accl_hw hw)
{
  std::string targetDevice;
  std::vector<std::string> strVector;
  std::vector<std::string>::iterator strVectorIter;

  if (this->_isLoaded) {
    /** @todo Can OpenVino support to replace the loaded model with a new one? */
    ml_loge ("The model file is already loaded onto the device.");
    return RetEBusy;
  }

  strVector = this->_ieCore.GetAvailableDevices ();
  if (strVector.size () == 0) {
    ml_loge ("No devices found for the OpenVino toolkit; "
             "check your plugin is installed, and the device is also connected.");
    return RetENoDev;
  }

  if (!TensorFilterOpenvino::isAcclDevSupported (strVector, hw)) {
    ml_loge ("Failed to find the device (%s) or its plugin (%s)",
        get_accl_hw_str (hw), _nnsAcclHwToOVDevMap[hw].c_str ());
    return RetEInval;
  }

#ifdef __OPENVINO_CPU_EXT__
  if (hw == ACCL_CPU) {
    this->_ieCore.AddExtension (
        std::make_shared<InferenceEngine::Extensions::Cpu::CpuExtensions> (),
        _nnsAcclHwToOVDevMap[hw]);
  }
#endif
  /** @todo Catch the IE exception */
  this->_executableNet
      = this->_ieCore.LoadNetwork (this->_networkCNN, _nnsAcclHwToOVDevMap[hw]);
  this->_hw = hw;
  this->_isLoaded = true;
  this->_inferRequest = this->_executableNet.CreateInferRequest ();

  return RetSuccess;
}

/**
 * @brief	Get the information about the dimensions of input tensors from the given model
 * @param[out] info metadata containing the dimesions and types information of the input tensors
 * @return 0 (TensorFilterOpenvino::RetSuccess) if OK, negative values if error
 */
int
TensorFilterOpenvino::getInputTensorDim (GstTensorsInfo *info)
{
  InferenceEngine::InputsDataMap *inputsDataMap = &(this->_inputsDataMap);
  InferenceEngine::InputsDataMap::iterator inputDataMapIter;
  int ret, i, j;

  gst_tensors_info_init (info);

  info->num_tensors = (uint32_t)inputsDataMap->size ();
  if (info->num_tensors > NNS_TENSOR_SIZE_LIMIT) {
    ml_loge ("The number of input tenosrs in the model "
             "exceeds more than NNS_TENSOR_SIZE_LIMIT, %s",
        NNS_TENSOR_SIZE_LIMIT_STR);
    ret = RetEOverFlow;
    goto failed;
  }

  for (inputDataMapIter = inputsDataMap->begin (), i = 0;
       inputDataMapIter != inputsDataMap->end (); ++inputDataMapIter, ++i) {
    InferenceEngine::SizeVector::reverse_iterator sizeVecRIter;
    InferenceEngine::TensorDesc eachInputTensorDesc;
    InferenceEngine::InputInfo::Ptr eachInputInfo;
    InferenceEngine::SizeVector dimsSizeVec;
    std::string ieTensorTypeStr;
    tensor_type nnsTensorType;

    eachInputInfo = inputDataMapIter->second;
    eachInputTensorDesc = eachInputInfo->getTensorDesc ();
    dimsSizeVec = eachInputTensorDesc.getDims ();
    if (dimsSizeVec.size () > NNS_TENSOR_RANK_LIMIT) {
      ml_loge ("The ranks of dimensions of InputTensor[%d] in the model "
               "exceeds NNS_TENSOR_RANK_LIMIT, %u",
          i, NNS_TENSOR_RANK_LIMIT);
      ret = RetEOverFlow;
      goto failed;
    }

    for (sizeVecRIter = dimsSizeVec.rbegin (), j = 0;
         sizeVecRIter != dimsSizeVec.rend (); ++sizeVecRIter, ++j) {
      info->info[i].dimension[j] = (*sizeVecRIter != 0 ? *sizeVecRIter : 1);
    }
    for (int k = j; k < NNS_TENSOR_RANK_LIMIT; ++k) {
      info->info[i].dimension[k] = 1;
    }

    ieTensorTypeStr = eachInputInfo->getPrecision ().name ();
    nnsTensorType = TensorFilterOpenvino::convertFromIETypeStr (ieTensorTypeStr);
    if (nnsTensorType == _NNS_END) {
      ml_loge ("The type of tensor elements, %s, "
               "in the model is not supported",
          ieTensorTypeStr.c_str ());
      ret = RetEInval;
      goto failed;
    }

    info->info[i].type = nnsTensorType;
    info->info[i].name = g_strdup (eachInputInfo->name ().c_str ());
    this->_inputTensorDescs[i] = eachInputTensorDesc;
  }

  return TensorFilterOpenvino::RetSuccess;

failed:
  ml_loge ("Failed to get dimension information about input tensor");

  return ret;
}

/**
 * @brief	Get the information about the dimensions of output tensors from the given model
 * @param[out] info metadata containing the dimesions and types information of the output tensors
 * @return 0 (TensorFilterOpenvino::RetSuccess) if OK, negative values if error
 */
int
TensorFilterOpenvino::getOutputTensorDim (GstTensorsInfo *info)
{
  InferenceEngine::OutputsDataMap *outputsDataMap = &(this->_outputsDataMap);
  InferenceEngine::OutputsDataMap::iterator outputDataMapIter;
  int ret, i, j;

  gst_tensors_info_init (info);

  info->num_tensors = (uint32_t)outputsDataMap->size ();
  if (info->num_tensors > NNS_TENSOR_SIZE_LIMIT) {
    ml_loge ("The number of output tenosrs in the model "
             "exceeds more than NNS_TENSOR_SIZE_LIMIT, %s",
        NNS_TENSOR_SIZE_LIMIT_STR);
    ret = RetEOverFlow;
    goto failed;
  }

  for (outputDataMapIter = outputsDataMap->begin (), i = 0;
       outputDataMapIter != outputsDataMap->end (); ++outputDataMapIter, ++i) {
    InferenceEngine::SizeVector::reverse_iterator sizeVecRIter;
    InferenceEngine::TensorDesc eachOutputTensorDesc;
    InferenceEngine::SizeVector dimsSizeVec;
    InferenceEngine::DataPtr eachOutputInfo;
    std::string ieTensorTypeStr;
    tensor_type nnsTensorType;

    eachOutputInfo = outputDataMapIter->second;
    eachOutputTensorDesc = eachOutputInfo->getTensorDesc ();
    dimsSizeVec = eachOutputTensorDesc.getDims ();
    if (dimsSizeVec.size () > NNS_TENSOR_RANK_LIMIT) {
      ml_loge ("The ranks of dimensions of OutputTensor[%d] in the model "
               "exceeds NNS_TENSOR_RANK_LIMIT, %u",
          i, NNS_TENSOR_RANK_LIMIT);
      ret = RetEOverFlow;
      goto failed;
    }

    for (sizeVecRIter = dimsSizeVec.rbegin (), j = 0;
         sizeVecRIter != dimsSizeVec.rend (); ++sizeVecRIter, ++j) {
      info->info[i].dimension[j] = (*sizeVecRIter != 0 ? *sizeVecRIter : 1);
    }
    for (int k = j; k < NNS_TENSOR_RANK_LIMIT; ++k) {
      info->info[i].dimension[k] = 1;
    }

    ieTensorTypeStr = eachOutputInfo->getPrecision ().name ();
    nnsTensorType = TensorFilterOpenvino::convertFromIETypeStr (ieTensorTypeStr);
    if (nnsTensorType == _NNS_END) {
      ml_loge ("The type of tensor elements, %s, "
               "in the model is not supported",
          ieTensorTypeStr.c_str ());
      ret = RetEInval;
      goto failed;
    }

    info->info[i].type = nnsTensorType;
    info->info[i].name = g_strdup (eachOutputInfo->getName ().c_str ());
    this->_outputTensorDescs[i] = eachOutputTensorDesc;
  }

  return TensorFilterOpenvino::RetSuccess;

failed:
  ml_loge ("Failed to get dimension information about output tensor");

  return ret;
}

/**
 * @brief Do inference using Inference Engine of the OpenVino framework
 * @param prop property of tensor_filter instance
 * @param[in] input the array of input tensors
 * @param[out] output the array of output tensors
 * @return RetSuccess if OK. non-zero if error
 */
int
TensorFilterOpenvino::invoke (const GstTensorFilterProperties *prop,
    const GstTensorMemory *input, GstTensorMemory *output)
{
  InferenceEngine::BlobMap inBlobMap;
  InferenceEngine::BlobMap outBlobMap;
  guint num_tensors;
  guint i;

  num_tensors = (prop->input_meta).num_tensors;
  for (i = 0; i < num_tensors; ++i) {
    const GstTensorInfo *info = &((prop->input_meta).info[i]);
    InferenceEngine::Blob::Ptr blob = convertGstTensorMemoryToBlobPtr (
        this->_inputTensorDescs[i], &(input[i]), prop->input_meta.info[i].type);
    if (blob == nullptr) {
      ml_loge ("Failed to create a blob for the input tensor: %u", i);
      return RetEInval;
    }
    inBlobMap.insert (make_pair (std::string (info->name), blob));
  }
  this->_inferRequest.SetInput (inBlobMap);

  num_tensors = (prop->output_meta).num_tensors;
  for (i = 0; i < num_tensors; ++i) {
    const GstTensorInfo *info = &((prop->output_meta).info[i]);
    InferenceEngine::Blob::Ptr blob = convertGstTensorMemoryToBlobPtr (
        this->_outputTensorDescs[i], &(output[i]), prop->output_meta.info[i].type);
    outBlobMap.insert (make_pair (std::string (info->name), blob));
    if (blob == nullptr) {
      ml_loge ("Failed to create a blob for the output tensor: %u", i);
      return RetEInval;
    }
  }
  this->_inferRequest.SetOutput (outBlobMap);

  this->_inferRequest.Infer ();

  return RetSuccess;
}

/**
 * @brief The mandatory callback for GstTensorFilterFramework
 * @param prop property of tensor_filter instance
 * @param private_data TensorFilterOpenvino plugin's private data
 * @param[in] input the array of input tensors
 * @param[out] output the array of output tensors
 * @return 0 if OK. non-zero if error
 */
static int
ov_invoke (const GstTensorFilterProperties *prop, void **private_data,
    const GstTensorMemory *input, GstTensorMemory *output)
{
  TensorFilterOpenvino *tfOv = static_cast<TensorFilterOpenvino *> (*private_data);

  return tfOv->invoke (prop, input, output);
}

/**
 * @brief The optional callback for GstTensorFilterFramework
 * @param prop property of tensor_filter instance
 * @param private_data TensorFilterOpenvino plugin's private data
 * @param[out] info the dimesions and types of input tensors
 * @return 0 (TensorFilterOpenvino::RetSuccess) if OK, negative values if error
 */
static int
ov_getInputDim (const GstTensorFilterProperties *prop, void **private_data, GstTensorsInfo *info)
{
  UNUSED (prop);
  TensorFilterOpenvino *tfOv = static_cast<TensorFilterOpenvino *> (*private_data);

  g_return_val_if_fail (tfOv != nullptr, TensorFilterOpenvino::RetEInval);

  return tfOv->getInputTensorDim (info);
}

/**
 * @brief The optional callback for GstTensorFilterFramework
 * @param prop property of tensor_filter instance
 * @param private_data TensorFilterOpenvino plugin's private data
 * @param[out] info The dimesions and types of output tensors
 * @return 0 (TensorFilterOpenvino::RetSuccess) if OK, negative values if error
 */
static int
ov_getOutputDim (const GstTensorFilterProperties *prop, void **private_data,
    GstTensorsInfo *info)
{
  UNUSED (prop);
  TensorFilterOpenvino *tfOv = static_cast<TensorFilterOpenvino *> (*private_data);

  g_return_val_if_fail (tfOv != nullptr, TensorFilterOpenvino::RetEInval);

  return tfOv->getOutputTensorDim (info);
}

/**
 * @brief Standard tensor_filter callback to close sub-plugin
 */
static void
ov_close (const GstTensorFilterProperties *prop, void **private_data)
{
  UNUSED (prop);
  TensorFilterOpenvino *tfOv = static_cast<TensorFilterOpenvino *> (*private_data);

  delete tfOv;
  *private_data = NULL;
}

/**
 * @brief Standard tensor_filter callback to open sub-plugin
 * @return 0 (TensorFilterOpenvino::RetSuccess) if OK, negative values if error
 */
static int
ov_open (const GstTensorFilterProperties *prop, void **private_data)
{
  std::string path_model_prefix;
  std::string model_path_xml;
  std::string model_path_bin;
  guint num_models_xml = 0;
  guint num_models_bin = 0;
  TensorFilterOpenvino *tfOv;
  accl_hw accelerator;

  accelerator = parse_accl_hw (prop->accl_str, openvino_accl_support, NULL, NULL);
#ifndef __OPENVINO_CPU_EXT__
  if (accelerator == ACCL_CPU) {
    ml_loge ("Accelerating via CPU is not supported on the current platform");
    return TensorFilterOpenvino::RetEInval;
  }
#endif
  if (accelerator == ACCL_NONE) {
    if (prop->accl_str != NULL) {
      ml_loge ("'%s' is not valid value for the 'accelerator' property", prop->accl_str);
    } else {
      ml_loge ("Invalid value for the 'accelerator' property");
    }
    ml_loge ("An acceptable format is as follows: 'true:[cpu|npu.movidius]'."
             "Note that 'cpu' is only for the x86_64 architecture.");

    return TensorFilterOpenvino::RetEInval;
  }

  if (prop->num_models == 1) {
    if (g_str_has_suffix (prop->model_files[0], TensorFilterOpenvino::extBin.c_str ())
        || g_str_has_suffix (prop->model_files[0], TensorFilterOpenvino::extXml.c_str ())) {
      std::string model_path = std::string (prop->model_files[0]);

      path_model_prefix = model_path.substr (0, model_path.length () - 4);
    } else {
      path_model_prefix = std::string (prop->model_files[0]);
    }

    model_path_xml = path_model_prefix + TensorFilterOpenvino::extXml;
    model_path_bin = path_model_prefix + TensorFilterOpenvino::extBin;
  } else {
    for (gint i = 0; i < prop->num_models; ++i) {
      if (g_str_has_suffix (prop->model_files[i], TensorFilterOpenvino::extXml.c_str ())) {
        num_models_xml++;
        model_path_xml = std::string (prop->model_files[i]);
      } else if (g_str_has_suffix (prop->model_files[i],
                     TensorFilterOpenvino::extBin.c_str ())) {
        num_models_bin++;
        model_path_bin = std::string (prop->model_files[i]);
      }

      if (num_models_xml > 1) {
        ml_loge ("Too many model files in a XML format are provided.");
        return TensorFilterOpenvino::RetEInval;
      } else if (num_models_bin > 1) {
        ml_loge ("Too many model files in a BIN format are provided.");
        return TensorFilterOpenvino::RetEInval;
      }
    }
  }

  if (!g_file_test (model_path_xml.c_str (), G_FILE_TEST_IS_REGULAR)) {
    ml_loge ("Failed to open the XML model file, %s", model_path_xml.c_str ());
    return TensorFilterOpenvino::RetEInval;
  }
  if (!g_file_test (model_path_bin.c_str (), G_FILE_TEST_IS_REGULAR)) {
    ml_loge ("Failed to open the BIN model file, %s", model_path_bin.c_str ());
    return TensorFilterOpenvino::RetEInval;
  }

  tfOv = static_cast<TensorFilterOpenvino *> (*private_data);
  if (tfOv != nullptr) {
    if (tfOv->isModelLoaded ()) {
      if ((tfOv->getPathModelBin () == model_path_bin)
          && (tfOv->getPathModelXml () == model_path_xml)) {
        return TensorFilterOpenvino::RetSuccess;
      }
    }

    ov_close (prop, private_data);
    tfOv = nullptr;
  }

  tfOv = new TensorFilterOpenvino (model_path_xml, model_path_bin);
  *private_data = tfOv;

  return tfOv->loadModel (accelerator);
}

/**
 * @brief The optional callback for GstTensorFilterFramework
 * @param[in] hw backend accelerator hardware
 * @return 0 if supported. -errno if not supported.
 */
static int
ov_checkAvailability (accl_hw hw)
{
  if (g_strv_contains (openvino_accl_support, get_accl_hw_str (hw)))
    return 0;

  return -ENOENT;
}

static gchar filter_subplugin_openvino[] = "openvino";

static GstTensorFilterFramework NNS_support_openvino = {.version = GST_TENSOR_FILTER_FRAMEWORK_V0,
  .open = ov_open,
  .close = ov_close,
  {.v0 = {
       .name = filter_subplugin_openvino,
       .allow_in_place = FALSE,
       .allocate_in_invoke = FALSE,
       .run_without_model = FALSE,
       .verify_model_path = FALSE,
       .statistics = nullptr,
       .invoke_NN = ov_invoke,
       .getInputDimension = ov_getInputDim,
       .getOutputDimension = ov_getOutputDim,
       .setInputDimension = nullptr,
       .destroyNotify = nullptr,
       .reloadModel = nullptr,
       .handleEvent = nullptr,
       .checkAvailability = ov_checkAvailability,
       .allocateInInvoke = nullptr,
   } } };

/**
 * @brief Initialize this object for tensor_filter sub-plugin runtime register
 */
void
init_filter_openvino (void)
{
  nnstreamer_filter_probe (&NNS_support_openvino);
}

/**
 * @brief Finalize the subplugin
 */
void
fini_filter_openvino (void)
{
  nnstreamer_filter_exit (NNS_support_openvino.v0.name);
}
