/* SPDX-License-Identifier: LGPL-2.1-only */
/**
 * GStreamer / NNStreamer gRPC/protobuf support
 * Copyright (C) 2020 Dongju Chae <dongju.chae@samsung.com>
 */
/**
 * @file    nnstreamer_grpc_protobuf.h
 * @date    21 Oct 2020
 * @brief   gRPC/Protobuf wrappers for nnstreamer
 * @see     https://github.com/nnstreamer/nnstreamer
 * @author  Dongju Chae <dongju.chae@samsung.com>
 * @bug     No known bugs except for NYI items
 */

#ifndef __NNS_GRPC_PROTOBUF_H__
#define __NNS_GRPC_PROTOBUF_H__

#include "nnstreamer_grpc_common.h"
#include "nnstreamer.grpc.pb.h" /* Generated by `protoc` */

using nnstreamer::protobuf::TensorService;
using nnstreamer::protobuf::Tensors;
using nnstreamer::protobuf::Tensor;

using google::protobuf::Empty;

namespace grpc {

/**
 * @brief NNStreamer gRPC protobuf service impl.
 */
class ServiceImplProtobuf : public NNStreamerRPC
{
  public:
    ServiceImplProtobuf (const grpc_config * config);

    void parse_tensors (Tensors &tensors);
    gboolean fill_tensors (Tensors &tensors);

  protected:
    template <typename T>
    grpc::Status _write_tensors (T writer);

    template <typename T>
    grpc::Status _read_tensors (T reader);

    void _get_tensors_from_buffer (GstBuffer *buffer, Tensors &tensors);
    void _get_buffer_from_tensors (Tensors &tensors, GstBuffer **buffer);

    std::unique_ptr<nnstreamer::protobuf::TensorService::Stub> client_stub_;
};

/**
 * @brief NNStreamer gRPC protobuf sync service impl.
 */
class SyncServiceImplProtobuf final
  : public ServiceImplProtobuf, public TensorService::Service
{
  public:
    SyncServiceImplProtobuf (const grpc_config * config);

    Status SendTensors (ServerContext *context, ServerReader<Tensors> *reader,
        Empty *reply) override;

    Status RecvTensors (ServerContext *context, const Empty *request,
        ServerWriter<Tensors> *writer) override;

  private:
    gboolean start_server (std::string address) override;
    gboolean start_client (std::string address) override;

    void _client_thread ();
};

class AsyncCallData;

/**
 * @brief NNStreamer gRPC protobuf async service impl.
 */
class AsyncServiceImplProtobuf final
  : public ServiceImplProtobuf, public TensorService::AsyncService
{
  public:
    AsyncServiceImplProtobuf (const grpc_config * config);
    ~AsyncServiceImplProtobuf ();

    /** @brief set the last call data */
    void set_last_call (AsyncCallData * call) { last_call_ = call; }

  private:
    gboolean start_server (std::string address) override;
    gboolean start_client (std::string address) override;

    void _server_thread ();
    void _client_thread ();

    AsyncCallData * last_call_;
};

/** @brief Internal base class to serve a request */
class AsyncCallData {
  public:
    /** @brief Constructor of AsyncCallData */
    AsyncCallData (AsyncServiceImplProtobuf *service)
      : service_ (service), state_ (CREATE), count_ (0)
    {
    }

    /** @brief Destructor of AsyncCallData */
    virtual ~AsyncCallData () {}

    /** @brief FSM-based state handling function */
    virtual void RunState (bool ok) {}

  protected:
    enum CallState { CREATE, PROCESS, FINISH, DESTROY };

    AsyncServiceImplProtobuf *service_;
    CallState state_;
    guint count_;

    Tensors rpc_tensors_;
    Empty rpc_empty_;
};

}; // namespace grpc

#endif /* __NNS_GRPC_PROTOBUF_H__ */
