/****************************************************************************************************************************************************
* Copyright (c) 2014 Freescale Semiconductor, Inc.
* All rights reserved.
*
* Redistribution and use in source and binary forms, with or without
* modification, are permitted provided that the following conditions are met:
*
*    * Redistributions of source code must retain the above copyright notice,
*      this list of conditions and the following disclaimer.
*
*    * Redistributions in binary form must reproduce the above copyright notice,
*      this list of conditions and the following disclaimer in the documentation
*      and/or other materials provided with the distribution.
*
*    * Neither the name of the Freescale Semiconductor, Inc. nor the names of
*      its contributors may be used to endorse or promote products derived from
*      this software without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
* ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
* WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
* IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
* INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
* BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
* DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
* LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
* OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
* ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
****************************************************************************************************************************************************/

#include <FslBase/Exceptions.hpp>
#include <FslBase/IO/Path.hpp>
#include <FslBase/Log/Log.hpp>
#include <FslDemoApp/Service/Persistent/IPersistentDataManager.hpp>
#include <FslDemoApp/Service/Graphics/IGraphicsService.hpp>
#include <FslDemoHost/Service/DemoAppControl/IDemoAppControlEx.hpp>
#include <FslDemoHost/Service/Test/TestService.hpp>
#include <cassert>
#include <iomanip>
#include <sstream>

namespace Fsl
{
  TestService::TestService(const ServiceProvider& serviceProvider)
    : ThreadLocalService(serviceProvider)
    , m_demoAppControlService()
    , m_persistentDataManager()
    , m_graphicsService()
    , m_screenshotNameScheme(TestScreenshotNameScheme::FrameNumber)
    , m_frameCounter(0)
    , m_screenshotNameCounter(0)
    , m_screenshotFrequency(0)
    , m_screenshotCount(0)
    , m_screenshot()
  {
    m_demoAppControlService = serviceProvider.Get<IDemoAppControlEx>();
    m_persistentDataManager = serviceProvider.Get<IPersistentDataManager>();
    m_graphicsService = serviceProvider.Get<IGraphicsService>();
  }


  TestService::~TestService()
  {
  }


  uint32_t TestService::GetScreenshotFrequency() const
  {
    return m_screenshotFrequency;
  }


  void TestService::SetScreenshotFrequency(const uint32_t frequency)
  {
    m_screenshotFrequency = frequency;
  }


  TestScreenshotNameScheme TestService::GetScreenshotNameScheme() const
  {
    return m_screenshotNameScheme;
  }


  void TestService::SetScreenshotNameScheme(const TestScreenshotNameScheme scheme)
  {
    m_screenshotNameScheme = scheme;
  }


  void TestService::OnFrameSwapCompleted()
  {
    ++m_frameCounter;
    if (m_screenshotNameScheme == TestScreenshotNameScheme::FrameNumber)
      ++m_screenshotNameCounter;

    const bool saveNow = (m_screenshotFrequency > 0 && (m_frameCounter % m_screenshotFrequency) == 0);
    const bool hasRequest = m_demoAppControlService->HasScreenshotRequest();

    if ( saveNow || hasRequest )
    {
      // Reset the update timer since the screenshot functionality is slow
      // We do this to allow a perfect timed capture of the frames
      m_demoAppControlService->RequestUpdateTimerReset();

      // Capture a screenshot and save it as a bmp
      m_graphicsService->Capture(m_screenshot, PixelFormat::B8G8R8A8_UINT);

      if (m_screenshot.IsValid())
      {
        if (saveNow)
        {
          std::stringstream stringStream;

          if (m_screenshotNameScheme == TestScreenshotNameScheme::Sequential)
          {
            ++m_screenshotNameCounter;
            stringStream << "test-screenshot-" << std::setw(10) << std::setfill('0') << m_screenshotNameCounter << ".bmp";
          }
          else
          {
            stringStream << "test-frame-" << std::setw(10) << std::setfill('0') << m_screenshotNameCounter << ".bmp";
          }

          m_persistentDataManager->Write(stringStream.str(), m_screenshot);
        }
        if (hasRequest)
        {
          m_demoAppControlService->ClearScreenshotRequestRequest();
          ++m_screenshotCount;
          std::stringstream stringStream;
          stringStream << "Screenshot-" << std::setw(10) << std::setfill('0') << m_screenshotCount << ".bmp";

          m_persistentDataManager->Write(stringStream.str(), m_screenshot);
        }
      }
    }
  }
}
