/*
 * Copyright (C) 2017, Boundary Devices <info@boundarydevices.com>
 *
 * SPDX-License-Identifier:	GPL-2.0+
 */

#include <common.h>
#include <asm/io.h>
#include <asm/arch/clock.h>
#include <asm/arch/imx-regs.h>
#include <asm/arch/iomux.h>
#include <asm/arch/sys_proto.h>
#include <malloc.h>
#include <asm/arch/mx6-pins.h>
#include <linux/errno.h>
#include <asm/gpio.h>
#include <asm/mach-imx/boot_mode.h>
#include <asm/mach-imx/fbpanel.h>
#include <asm/mach-imx/iomux-v3.h>
#include <asm/mach-imx/mxc_i2c.h>
#include <asm/mach-imx/spi.h>
#include <mmc.h>
#include <fsl_esdhc_imx.h>
#include <linux/fb.h>
#include <ipu_pixfmt.h>
#include <asm/arch/crm_regs.h>
#include <asm/arch/mxc_hdmi.h>
#include <i2c.h>
#include <input.h>
#include <pwm.h>
#include <splash.h>
#include <usb/ehci-ci.h>
#include "../common/bd_common.h"
#include "../common/padctrl.h"

/* Special MXCFB sync flags are here. */
#include "../drivers/video/mxcfb.h"

DECLARE_GLOBAL_DATA_PTR;

#define AUD_PAD_CTRL  (PAD_CTL_PUS_100K_UP |			\
	PAD_CTL_SPEED_LOW | PAD_CTL_DSE_40ohm |			\
	PAD_CTL_HYS | PAD_CTL_SRE_FAST)

#define I2C_PAD_CTRL	(PAD_CTL_PUS_100K_UP |			\
	PAD_CTL_SPEED_MED | PAD_CTL_DSE_40ohm | PAD_CTL_HYS |	\
	PAD_CTL_ODE | PAD_CTL_SRE_FAST)

#define RGB_PAD_CTRL	PAD_CTL_DSE_120ohm

#define SPI_PAD_CTRL (PAD_CTL_HYS |				\
	PAD_CTL_PUS_100K_DOWN | PAD_CTL_SPEED_MED |		\
	PAD_CTL_DSE_40ohm | PAD_CTL_SRE_FAST)

#define UART_PAD_CTRL  (PAD_CTL_PUS_100K_UP |			\
	PAD_CTL_SPEED_MED | PAD_CTL_DSE_40ohm |			\
	PAD_CTL_SRE_FAST | PAD_CTL_HYS)

#define USDHC_PAD_CTRL (PAD_CTL_PUS_47K_UP |			\
	PAD_CTL_SPEED_LOW | PAD_CTL_DSE_40ohm |			\
	PAD_CTL_SRE_FAST | PAD_CTL_HYS)

static const iomux_v3_cfg_t init_pads[] = {
	/* Audmux 3 - Goes to the WM8960 */
	IOMUX_PAD_CTRL(CSI0_DAT4__AUD3_TXC, AUD_PAD_CTRL),
	IOMUX_PAD_CTRL(CSI0_DAT5__AUD3_TXD, AUD_PAD_CTRL),
	IOMUX_PAD_CTRL(CSI0_DAT6__AUD3_TXFS, AUD_PAD_CTRL),
	IOMUX_PAD_CTRL(CSI0_DAT7__AUD3_RXD, AUD_PAD_CTRL),

	/* Audmux 4 - Goes to the WWAN USB Modem */
	IOMUX_PAD_CTRL(DISP0_DAT20__AUD4_TXC, AUD_PAD_CTRL),
	IOMUX_PAD_CTRL(DISP0_DAT21__AUD4_TXD, AUD_PAD_CTRL),
	IOMUX_PAD_CTRL(DISP0_DAT22__AUD4_TXFS, AUD_PAD_CTRL),
	IOMUX_PAD_CTRL(DISP0_DAT23__AUD4_RXD, AUD_PAD_CTRL),

	/* backlight lvds */
#define GP_LVDS_BKL_EN		IMX_GPIO_NR(1, 2)
	IOMUX_PAD_CTRL(GPIO_2__GPIO1_IO02, WEAK_PULLDN),
#define GP_BACKLIGHT_LVDS_PWM1	IMX_GPIO_NR(1, 21)
	IOMUX_PAD_CTRL(SD1_DAT3__GPIO1_IO21, WEAK_PULLDN),
#define GP_LVDS_LED_EN		IMX_GPIO_NR(4, 19)
	IOMUX_PAD_CTRL(DI0_PIN3__GPIO4_IO19, WEAK_PULLDN),
#define GP_LVDS_BIST		IMX_GPIO_NR(5, 9)
	IOMUX_PAD_CTRL(DISP0_DAT15__GPIO5_IO09, WEAK_PULLDN),

	/* bt_rfkill */
#define GP_BT_RFKILL_RESET	IMX_GPIO_NR(6, 16)
	IOMUX_PAD_CTRL(NANDF_CS3__GPIO6_IO16, WEAK_PULLDN),

	/* ECSPI1 */
	IOMUX_PAD_CTRL(EIM_D17__ECSPI1_MISO, SPI_PAD_CTRL),
	IOMUX_PAD_CTRL(EIM_D18__ECSPI1_MOSI, SPI_PAD_CTRL),
	IOMUX_PAD_CTRL(EIM_D16__ECSPI1_SCLK, SPI_PAD_CTRL),
#define GP_ECSPI1_NOR_CS	IMX_GPIO_NR(3, 19)
	IOMUX_PAD_CTRL(EIM_D19__GPIO3_IO19, SPI_PAD_CTRL),

	/* ECSPI2 - J6, finger sensor */
	IOMUX_PAD_CTRL(CSI0_DAT10__ECSPI2_MISO, SPI_PAD_CTRL),	/* pin 3 */
	IOMUX_PAD_CTRL(CSI0_DAT9__ECSPI2_MOSI, SPI_PAD_CTRL),	/* pin 4 */
	IOMUX_PAD_CTRL(CSI0_DAT8__ECSPI2_SCLK, SPI_PAD_CTRL),	/* pin 5 */
#define GP_ECSPI2_CS		IMX_GPIO_NR(5, 29)
	IOMUX_PAD_CTRL(CSI0_DAT11__GPIO5_IO29, SPI_PAD_CTRL),	/* pin 6 */

	/* ENET pads that don't change for PHY reset */
	IOMUX_PAD_CTRL(ENET_MDIO__ENET_MDIO, PAD_CTRL_ENET_MDIO),
	IOMUX_PAD_CTRL(ENET_MDC__ENET_MDC, PAD_CTRL_ENET_MDC),
	IOMUX_PAD_CTRL(RGMII_TXC__RGMII_TXC, PAD_CTRL_ENET_TX),
	IOMUX_PAD_CTRL(RGMII_TD0__RGMII_TD0, PAD_CTRL_ENET_TX),
	IOMUX_PAD_CTRL(RGMII_TD1__RGMII_TD1, PAD_CTRL_ENET_TX),
	IOMUX_PAD_CTRL(RGMII_TD2__RGMII_TD2, PAD_CTRL_ENET_TX),
	IOMUX_PAD_CTRL(RGMII_TD3__RGMII_TD3, PAD_CTRL_ENET_TX),
	IOMUX_PAD_CTRL(RGMII_TX_CTL__RGMII_TX_CTL, PAD_CTRL_ENET_TX),
	IOMUX_PAD_CTRL(ENET_REF_CLK__ENET_TX_CLK, PAD_CTRL_ENET_TX),
	/* pin 1 nRST of AR8035 */
#define GP_RGMII_PHY_RESET	IMX_GPIO_NR(1, 27)
	IOMUX_PAD_CTRL(ENET_RXD0__GPIO1_IO27, WEAK_PULLDN),
#define GPIRQ_ENET_PHY		IMX_GPIO_NR(1, 28)
	IOMUX_PAD_CTRL(ENET_TX_EN__GPIO1_IO28, WEAK_PULLUP),

#define GP_GPIOKEY_POWER	IMX_GPIO_NR(3, 1)
	IOMUX_PAD_CTRL(EIM_DA1__GPIO3_IO01, WEAK_PULLUP),
#define GP_GPIOKEY_VOL_DN	IMX_GPIO_NR(3, 5)
	IOMUX_PAD_CTRL(EIM_DA5__GPIO3_IO05, WEAK_PULLUP),
#define GP_GPIOKEY_VOL_UP	IMX_GPIO_NR(3, 6)
	IOMUX_PAD_CTRL(EIM_DA6__GPIO3_IO06, WEAK_PULLUP),

	/* Hogs */
#define GP_GPIO_A		IMX_GPIO_NR(3, 20)
	IOMUX_PAD_CTRL(EIM_D20__GPIO3_IO20, WEAK_PULLUP),
#define GP_GPIO_B		IMX_GPIO_NR(4, 21)
	IOMUX_PAD_CTRL(DISP0_DAT0__GPIO4_IO21, WEAK_PULLUP),

#define GP_STAT_LED1		IMX_GPIO_NR(4, 26)		/* Blue */
	IOMUX_PAD_CTRL(DISP0_DAT5__GPIO4_IO26, WEAK_PULLUP),
#define GP_STAT_LED2		IMX_GPIO_NR(4, 25)		/* Greeen */
	IOMUX_PAD_CTRL(DISP0_DAT4__GPIO4_IO25, WEAK_PULLUP),
#define GP_STAT_LED3		IMX_GPIO_NR(4, 27)		/* Red */
	IOMUX_PAD_CTRL(DISP0_DAT6__GPIO4_IO27, WEAK_PULLUP),

	/* Hogs - FAN48623, regulator for WWAN usb modem, always on for now */
#define GP_3P7_EN		IMX_GPIO_NR(5, 11)
	IOMUX_PAD_CTRL(DISP0_DAT17__GPIO5_IO11, WEAK_PULLDN),	/* active high */
#define GP_3P7_BYPASS_EN	IMX_GPIO_NR(5, 0)
	IOMUX_PAD_CTRL(EIM_WAIT__GPIO5_IO00, WEAK_PULLUP),	/* low active */

#define GPIRQ_TAMPER	IMX_GPIO_NR(4, 18)
	IOMUX_PAD_CTRL(DI0_PIN2__GPIO4_IO18, WEAK_PULLDN),
#define GPIRQ_TAMPER2	IMX_GPIO_NR(5, 12)
	IOMUX_PAD_CTRL(DISP0_DAT18__GPIO5_IO12, WEAK_PULLDN),

	/* TODO Wireless control pins */
#define GP_BT_CLK_REQ		IMX_GPIO_NR(6, 8)
	IOMUX_PAD_CTRL(NANDF_ALE__GPIO6_IO08, WEAK_PULLUP),
#define GP_BT_HOST_WAKE		IMX_GPIO_NR(6, 7)
	IOMUX_PAD_CTRL(NANDF_CLE__GPIO6_IO07, WEAK_PULLUP),
#define GP_WIFI_QOW		IMX_GPIO_NR(2, 3)
	IOMUX_PAD_CTRL(NANDF_D3__GPIO2_IO03, WEAK_PULLUP),

#define GP_TP68			IMX_GPIO_NR(2, 0)
	IOMUX_PAD_CTRL(NANDF_D0__GPIO2_IO00, WEAK_PULLUP),
#define GP_TP71			IMX_GPIO_NR(1, 30)
	IOMUX_PAD_CTRL(ENET_TXD0__GPIO1_IO30, WEAK_PULLUP),
#define GP_TP72			IMX_GPIO_NR(2, 4)
	IOMUX_PAD_CTRL(NANDF_D4__GPIO2_IO04, WEAK_PULLUP),
#define GP_TP73			IMX_GPIO_NR(2, 2)
	IOMUX_PAD_CTRL(NANDF_D2__GPIO2_IO02, WEAK_PULLUP),
#define GP_TP74			IMX_GPIO_NR(2, 7)
	IOMUX_PAD_CTRL(NANDF_D7__GPIO2_IO07, WEAK_PULLDN),
#define GP_TP78			IMX_GPIO_NR(6, 9)
	IOMUX_PAD_CTRL(NANDF_WP_B__GPIO6_IO09, WEAK_PULLUP),
#define GP_TP85			IMX_GPIO_NR(1, 9)
	IOMUX_PAD_CTRL(GPIO_9__GPIO1_IO09, WEAK_PULLUP),
#define GP_TP86			IMX_GPIO_NR(4, 31)
	IOMUX_PAD_CTRL(DISP0_DAT10__GPIO4_IO31, WEAK_PULLUP),
#define GP_TP90			IMX_GPIO_NR(4, 17)
	IOMUX_PAD_CTRL(DI0_PIN15__GPIO4_IO17, WEAK_PULLUP),
#define GP_TP91			IMX_GPIO_NR(1, 16)
	IOMUX_PAD_CTRL(SD1_DAT0__GPIO1_IO16, WEAK_PULLUP),
#define GP_TP92			IMX_GPIO_NR(4, 20)
	IOMUX_PAD_CTRL(DI0_PIN4__GPIO4_IO20, WEAK_PULLUP),
#define GP_TP97			IMX_GPIO_NR(2, 30)
	IOMUX_PAD_CTRL(EIM_EB2__GPIO2_IO30, WEAK_PULLUP),
#define GP_TP104		IMX_GPIO_NR(3, 7)
	IOMUX_PAD_CTRL(EIM_DA7__GPIO3_IO07, WEAK_PULLUP),
#define GP_TP131		IMX_GPIO_NR(5, 2)
	IOMUX_PAD_CTRL(EIM_A25__GPIO5_IO02, WEAK_PULLUP),
#define GP_TP_R368		IMX_GPIO_NR(3, 30)
	IOMUX_PAD_CTRL(EIM_D30__GPIO3_IO30, WEAK_PULLUP),

	/* I2C1 - mux */
#define GP_I2C1MUX_A0		IMX_GPIO_NR(3, 15)
	IOMUX_PAD_CTRL(EIM_DA15__GPIO3_IO15, WEAK_PULLDN),
#define GP_I2C1MUX_A1		IMX_GPIO_NR(3, 14)
	IOMUX_PAD_CTRL(EIM_DA14__GPIO3_IO14, WEAK_PULLDN),
#define GP_I2C1MUX_RESET	IMX_GPIO_NR(3, 13)
	IOMUX_PAD_CTRL(EIM_DA13__GPIO3_IO13, WEAK_PULLDN),

	/* I2C1(mux 0) - wm8960 */
	IOMUX_PAD_CTRL(GPIO_0__CCM_CLKO1, WEAK_PULLDN),
#define GP_WM8960_MIC_DET		IMX_GPIO_NR(7, 8)
	IOMUX_PAD_CTRL(SD3_RST__GPIO7_IO08, WEAK_PULLUP),
#define GP_WM8960_HP_DET		IMX_GPIO_NR(4, 10)
	IOMUX_PAD_CTRL(KEY_COL2__GPIO4_IO10, WEAK_PULLUP),

	/* I2C1(mux 1) - accelerometer */
#define GPIRQ_MPU9250_INT	IMX_GPIO_NR(6, 11)
	IOMUX_PAD_CTRL(NANDF_CS0__GPIO6_IO11, WEAK_PULLUP),

	/* I2C1(mux 2)camera CSI0 J6  - finger sensor */
	IOMUX_PAD_CTRL(CSI0_DAT12__IPU1_CSI0_DATA12, WEAK_PULLUP),	/* pin 20 */
	IOMUX_PAD_CTRL(CSI0_DAT13__IPU1_CSI0_DATA13, WEAK_PULLUP),	/* pin 21 */
	IOMUX_PAD_CTRL(CSI0_DAT14__IPU1_CSI0_DATA14, WEAK_PULLUP),	/* pin 22 */
	IOMUX_PAD_CTRL(CSI0_DAT15__IPU1_CSI0_DATA15, WEAK_PULLUP),	/* pin 23 */
	IOMUX_PAD_CTRL(CSI0_DAT16__IPU1_CSI0_DATA16, WEAK_PULLUP),	/* pin 25 */
	IOMUX_PAD_CTRL(CSI0_DAT17__IPU1_CSI0_DATA17, WEAK_PULLUP),	/* pin 26 */
	IOMUX_PAD_CTRL(CSI0_DAT18__IPU1_CSI0_DATA18, WEAK_PULLUP),	/* pin 27 */
	IOMUX_PAD_CTRL(CSI0_DAT19__IPU1_CSI0_DATA19, WEAK_PULLUP),	/* pin 28 */
	IOMUX_PAD_CTRL(CSI0_DATA_EN__IPU1_CSI0_DATA_EN, WEAK_PULLUP),	/* pin 11 NS */
	IOMUX_PAD_CTRL(CSI0_PIXCLK__IPU1_CSI0_PIXCLK, WEAK_PULLUP),	/* pin 10 */
	IOMUX_PAD_CTRL(CSI0_MCLK__IPU1_CSI0_HSYNC, WEAK_PULLUP),	/* pin 16 */
	IOMUX_PAD_CTRL(CSI0_VSYNC__IPU1_CSI0_VSYNC, WEAK_PULLUP),	/* pin 17 */
#define GP_FP_RESET_N		IMX_GPIO_NR(1, 8)
	IOMUX_PAD_CTRL(GPIO_8__GPIO1_IO08, WEAK_PULLDN),		/* pin 18 */
#define GP_FP_SENSOR_STAT	IMX_GPIO_NR(2, 27)
	IOMUX_PAD_CTRL(EIM_LBA__GPIO2_IO27, WEAK_PULLUP),	/* pin 2 */
#define GP_FP_LE_EN		IMX_GPIO_NR(2, 26)
	IOMUX_PAD_CTRL(EIM_RW__GPIO2_IO26, WEAK_PULLDN),		/* pin 11 */

	/* I2C1(mux 3) - ov5640 mipi */
	IOMUX_PAD_CTRL(GPIO_3__CCM_CLKO2, OUTPUT_40OHM),	/* mclk */
#define GP_OV5640_MIPI_POWER_DOWN	IMX_GPIO_NR(6, 10)
	IOMUX_PAD_CTRL(NANDF_RB0__GPIO6_IO10, WEAK_PULLUP),
#define GP_OV5640_MIPI_RESET		IMX_GPIO_NR(2, 5)
	IOMUX_PAD_CTRL(NANDF_D5__GPIO2_IO05, WEAK_PULLDN),

	/* I2C2, J1 - Front camera, ov5640 */
	IOMUX_PAD_CTRL(EIM_EB1__IPU2_CSI1_DATA10, WEAK_PULLUP),	/* pin 37 */
	IOMUX_PAD_CTRL(EIM_EB0__IPU2_CSI1_DATA11, WEAK_PULLUP),	/* pin 35 */
	/* pin 34, mclk, GPIO_3, also mipi_mclk */
	IOMUX_PAD_CTRL(EIM_A17__IPU2_CSI1_DATA12, WEAK_PULLUP),	/* pin 33 */
	IOMUX_PAD_CTRL(EIM_A18__IPU2_CSI1_DATA13, WEAK_PULLUP),	/* pin 31 */
	IOMUX_PAD_CTRL(EIM_A19__IPU2_CSI1_DATA14, WEAK_PULLUP),	/* pin 29 */
	IOMUX_PAD_CTRL(EIM_A20__IPU2_CSI1_DATA15, WEAK_PULLUP),	/* pin 27 */
	IOMUX_PAD_CTRL(EIM_A21__IPU2_CSI1_DATA16, WEAK_PULLUP),	/* pin 25 */
	IOMUX_PAD_CTRL(EIM_A22__IPU2_CSI1_DATA17, WEAK_PULLUP),	/* pin 23 */
	IOMUX_PAD_CTRL(EIM_A23__IPU2_CSI1_DATA18, WEAK_PULLUP),	/* pin 21 */
	IOMUX_PAD_CTRL(EIM_A24__IPU2_CSI1_DATA19, WEAK_PULLUP),	/* pin 19 */
	IOMUX_PAD_CTRL(EIM_DA10__IPU2_CSI1_DATA_EN, WEAK_PULLUP),	/* pin 14 */
	IOMUX_PAD_CTRL(EIM_A16__IPU2_CSI1_PIXCLK, WEAK_PULLUP),	/* pin 11 */
	IOMUX_PAD_CTRL(EIM_DA11__IPU2_CSI1_HSYNC, WEAK_PULLUP),	/* pin 15 */
	IOMUX_PAD_CTRL(EIM_DA12__IPU2_CSI1_VSYNC, WEAK_PULLUP),	/* pin 13 */
#define GP_OV5640_STROBE		IMX_GPIO_NR(2, 23)
	IOMUX_PAD_CTRL(EIM_CS0__GPIO2_IO23, WEAK_PULLUP),	/* pin 3 */
#define GP_OV5640_POWER_DOWN		IMX_GPIO_NR(2, 24)
	IOMUX_PAD_CTRL(EIM_CS1__GPIO2_IO24, WEAK_PULLUP),	/* pin 17 */
#define GP_OV5640_RESET			IMX_GPIO_NR(2, 25)
	IOMUX_PAD_CTRL(EIM_OE__GPIO2_IO25, WEAK_PULLDN),	/* pin 9 */

	/* I2C2 - GT928 Touchscreen */
#define GPIRQ_GT928	IMX_GPIO_NR(4, 15)
	IOMUX_PAD_CTRL(KEY_ROW4__GPIO4_IO15, WEAK_PULLUP),
#define GP_GT928_RESET	IMX_GPIO_NR(1, 7)
	IOMUX_PAD_CTRL(GPIO_7__GPIO1_IO07, WEAK_PULLDN),

	/* I2C2 - light sensor */
#define GPIRQ_LIGHTSENSOR	IMX_GPIO_NR(3, 0)
	IOMUX_PAD_CTRL(EIM_DA0__GPIO3_IO00, WEAK_PULLUP),

	/* I2C2 - RTC */
#define GPIRQ_RTC	IMX_GPIO_NR(1, 4)
	IOMUX_PAD_CTRL(GPIO_4__GPIO1_IO04, WEAK_PULLUP),

	/* I2C3 - LM3643 */
#define GP_FLASH_STROBE		IMX_GPIO_NR(2, 31)
	IOMUX_PAD_CTRL(EIM_EB3__GPIO2_IO31, WEAK_PULLDN),
#define GP_FLASH_HW_EN		IMX_GPIO_NR(5, 6)
	IOMUX_PAD_CTRL(DISP0_DAT12__GPIO5_IO06, WEAK_PULLDN),
#define GP_TORCH_EN		IMX_GPIO_NR(5, 5)
	IOMUX_PAD_CTRL(DISP0_DAT11__GPIO5_IO05, WEAK_PULLDN),
#define GP_FLASH_TX		IMX_GPIO_NR(4, 30)
	IOMUX_PAD_CTRL(DISP0_DAT9__GPIO4_IO30, WEAK_PULLDN),

	/* I2C3 - max77818 */
#define GPIRQ_MAX77818_INOK	IMX_GPIO_NR(5, 13)
	IOMUX_PAD_CTRL(DISP0_DAT19__GPIO5_IO13, WEAK_PULLUP),
#define GPIRQ_MAX77818_WCINOK	IMX_GPIO_NR(3, 2)
	IOMUX_PAD_CTRL(EIM_DA2__GPIO3_IO02, WEAK_PULLUP),
#define GPIRQ_MAX77818		IMX_GPIO_NR(3, 4)
	IOMUX_PAD_CTRL(EIM_DA4__GPIO3_IO04, WEAK_PULLUP),

	/* leds - Vibrator */
#define GP_LED_VIBRATOR		IMX_GPIO_NR(5, 8)
	IOMUX_PAD_CTRL(DISP0_DAT14__GPIO5_IO08, WEAK_PULLDN),

	/* PWM3, backlight (LVDS display) */
#define GP_LVDS_LED_PWM		IMX_GPIO_NR(1, 17)
	IOMUX_PAD_CTRL(SD1_DAT1__GPIO1_IO17, WEAK_PULLDN),

	/* PWM2, FP_MCLK */
#define GP_PWM2			IMX_GPIO_NR(1, 19)
	IOMUX_PAD_CTRL(SD1_DAT2__GPIO1_IO19, WEAK_PULLDN),

	/* PWM4, SIM_CLK  */
#define GP_SIM_CLK		IMX_GPIO_NR(1, 18)
	IOMUX_PAD_CTRL(SD1_CMD__GPIO1_IO18, WEAK_PULLUP),

	/* Regulators */
/* Goes to  J1(Front camera, EIM1), J5(mipi, rear), and NVCC_CSI/EIM1 */
#define GP_REG_2P8V_EN		IMX_GPIO_NR(3, 9)
	IOMUX_PAD_CTRL(EIM_DA9__GPIO3_IO09, WEAK_PULLDN),

#define GP_REG_FP_3P3V_EN	IMX_GPIO_NR(3, 29)
	IOMUX_PAD_CTRL(EIM_D29__GPIO3_IO29, WEAK_PULLDN),	/* J6 pin 7, 3.3V */

#define GP_REG_FP_5V_EN		IMX_GPIO_NR(6, 31)
	IOMUX_PAD_CTRL(EIM_BCLK__GPIO6_IO31, WEAK_PULLDN),	/* J6 pin 8, 5v */

	/* reg_usbotg_vbus */
#define GP_REG_USBOTG		IMX_GPIO_NR(3, 22)
	IOMUX_PAD_CTRL(EIM_D22__GPIO3_IO22, WEAK_PULLDN),

#define GP_REG_VBUS4		IMX_GPIO_NR(4, 11)
	IOMUX_PAD_CTRL(KEY_ROW2__GPIO4_IO11, WEAK_PULLDN),

	/* reg_wlan_en */
#define GP_REG_WLAN_EN		IMX_GPIO_NR(6, 15)
	IOMUX_PAD_CTRL(NANDF_CS2__GPIO6_IO15, WEAK_PULLDN),

	/* UART1 */
	IOMUX_PAD_CTRL(SD3_DAT7__UART1_TX_DATA, UART_PAD_CTRL),
	IOMUX_PAD_CTRL(SD3_DAT6__UART1_RX_DATA, UART_PAD_CTRL),

	/* UART2 */
	IOMUX_PAD_CTRL(EIM_D26__UART2_TX_DATA, UART_PAD_CTRL),
	IOMUX_PAD_CTRL(EIM_D27__UART2_RX_DATA, UART_PAD_CTRL),

	/* UART3 for BT */
	IOMUX_PAD_CTRL(EIM_D24__UART3_TX_DATA, UART_PAD_CTRL),
	IOMUX_PAD_CTRL(EIM_D25__UART3_RX_DATA, UART_PAD_CTRL),
	IOMUX_PAD_CTRL(EIM_D23__UART3_CTS_B, UART_PAD_CTRL),
	IOMUX_PAD_CTRL(EIM_D31__UART3_RTS_B, UART_PAD_CTRL),

	/* UART4 - J65 */
	IOMUX_PAD_CTRL(KEY_COL0__UART4_TX_DATA, UART_PAD_CTRL),
	IOMUX_PAD_CTRL(KEY_ROW0__UART4_RX_DATA, UART_PAD_CTRL),

	/* UART5 - GPS Ublox MAX-7W */
	IOMUX_PAD_CTRL(KEY_COL1__UART5_TX_DATA, UART_PAD_CTRL),
	IOMUX_PAD_CTRL(KEY_ROW1__UART5_RX_DATA, UART_PAD_CTRL),
#define GP_GPS_HEARTBEAT	IMX_GPIO_NR(7, 1)
	IOMUX_PAD_CTRL(SD3_DAT4__GPIO7_IO01, WEAK_PULLUP),
#define GP_GPS_INT		IMX_GPIO_NR(7, 13)
	IOMUX_PAD_CTRL(GPIO_18__GPIO7_IO13, WEAK_PULLUP),
#define GP_GPS_RESET		IMX_GPIO_NR(4, 5)
	IOMUX_PAD_CTRL(GPIO_19__GPIO4_IO05, WEAK_PULLDN),

	/* USBH1 */
#define GP_USBH1_HUB_RESET	IMX_GPIO_NR(7, 12)
	IOMUX_PAD_CTRL(GPIO_17__GPIO7_IO12, WEAK_PULLDN),

	/* USBOTG */
	IOMUX_PAD_CTRL(GPIO_1__USB_OTG_ID, WEAK_PULLUP),
	IOMUX_PAD_CTRL(KEY_COL4__USB_OTG_OC, WEAK_PULLUP),

	/* USDHC2 - Wifi */
	IOMUX_PAD_CTRL(SD2_CLK__SD2_CLK, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD2_CMD__SD2_CMD, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD2_DAT0__SD2_DATA0, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD2_DAT1__SD2_DATA1, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD2_DAT2__SD2_DATA2, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD2_DAT3__SD2_DATA3, USDHC_PAD_CTRL),
#define GPIRQ_WIFI		IMX_GPIO_NR(6, 14)
	IOMUX_PAD_CTRL(NANDF_CS1__GPIO6_IO14, WEAK_PULLDN),
#define GP_WIFI_WAKE		IMX_GPIO_NR(2, 1)
	IOMUX_PAD_CTRL(NANDF_D1__GPIO2_IO01, WEAK_PULLUP),
//	IOMUX_PAD_CTRL(SD1_CLK__OSC32K_32K_OUT, OUTPUT_40OHM),	/* slow clock */

	/* USDHC3 - SD card */
	IOMUX_PAD_CTRL(SD3_CLK__SD3_CLK, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD3_CMD__SD3_CMD, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD3_DAT0__SD3_DATA0, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD3_DAT1__SD3_DATA1, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD3_DAT2__SD3_DATA2, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD3_DAT3__SD3_DATA3, USDHC_PAD_CTRL),
#define GP_USDHC3_CD		IMX_GPIO_NR(7, 0)
	IOMUX_PAD_CTRL(SD3_DAT5__GPIO7_IO00, WEAK_PULLUP),

	/* USDHC4 - emmc */
	IOMUX_PAD_CTRL(SD4_CLK__SD4_CLK, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD4_CMD__SD4_CMD, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD4_DAT0__SD4_DATA0, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD4_DAT1__SD4_DATA1, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD4_DAT2__SD4_DATA2, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD4_DAT3__SD4_DATA3, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD4_DAT4__SD4_DATA4, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD4_DAT5__SD4_DATA5, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD4_DAT6__SD4_DATA6, USDHC_PAD_CTRL),
	IOMUX_PAD_CTRL(SD4_DAT7__SD4_DATA7, USDHC_PAD_CTRL),
#define GP_EMMC_RESET		IMX_GPIO_NR(2, 6)
	IOMUX_PAD_CTRL(NANDF_D6__GPIO2_IO06, WEAK_PULLUP),

	/* WWAN USB Modem */
#define GP_MODEM_RESET		IMX_GPIO_NR(4, 23)
	IOMUX_PAD_CTRL(DISP0_DAT2__GPIO4_IO23, WEAK_PULLDN),
#define GP_MODEM_PULSE_ON	IMX_GPIO_NR(4, 29)
	IOMUX_PAD_CTRL(DISP0_DAT8__GPIO4_IO29, WEAK_PULLDN),
#define GP_SLEEP_STAT		IMX_GPIO_NR(4, 22)
	IOMUX_PAD_CTRL(DISP0_DAT1__GPIO4_IO22, WEAK_PULLUP),
#define GP_SIM_DETECT		IMX_GPIO_NR(3, 8)
	IOMUX_PAD_CTRL(EIM_DA8__GPIO3_IO08, WEAK_PULLUP),
#define GP_SIM_IO		IMX_GPIO_NR(5, 10)
	IOMUX_PAD_CTRL(DISP0_DAT16__GPIO5_IO10, WEAK_PULLUP),
};

static const iomux_v3_cfg_t lvds_led_pwm_pads[] = {
	IOMUX_PAD_CTRL(SD1_DAT1__PWM3_OUT, WEAK_PULLDN),
};

static const iomux_v3_cfg_t lvds_led_pwm_gpio_pads[] = {
	IOMUX_PAD_CTRL(SD1_DAT1__GPIO1_IO17, WEAK_PULLDN),
};

static const struct i2c_pads_info i2c_pads[] = {
	/*
	 * pca9546 mux(0x70), J7 lvds,
	 *    wm8960(0x1a),
	 *    mpu9250(0x69),
	 *    finger-sensor(0x37),
	 *    ov5640_mipi(0x3c)
	 */
	I2C_PADS_INFO_ENTRY(I2C1, EIM_D21, 3, 21, EIM_D28, 3, 28, I2C_PAD_CTRL),
	/*
	 * gt928(0x14), lightsensor(0x29), ov5640(0x3c), rv4162(0x68)
	 */
	I2C_PADS_INFO_ENTRY(I2C2, KEY_COL3, 4, 12, KEY_ROW3, 4, 13, I2C_PAD_CTRL),
	/*
	 * fan53526(0x60), lm3643(0x63), max77818(0x66)
	 */
	I2C_PADS_INFO_ENTRY(I2C3, GPIO_5, 1, 05, GPIO_16, 7, 11, I2C_PAD_CTRL),
};
#define I2C_BUS_CNT	3

#ifdef CONFIG_USB_EHCI_MX6
int board_ehci_hcd_init(int port)
{
	if (port) {
		/* Reset USB hub */
		gpio_direction_output(GP_USBH1_HUB_RESET, 0);
		mdelay(2);
		gpio_set_value(GP_USBH1_HUB_RESET, 1);
	}
	return 0;
}

int board_ehci_power(int port, int on)
{
	int gp = port ? GP_REG_VBUS4 : GP_REG_USBOTG;

	if (on)
		max77823_boost_power(on);
	gpio_set_value(gp, on);
	return 0;
}
#endif

#ifdef CONFIG_FSL_ESDHC_IMX
struct fsl_esdhc_cfg board_usdhc_cfg[] = {
	{.esdhc_base = USDHC3_BASE_ADDR, .bus_width = 4,
			.gp_cd = GP_USDHC3_CD},
	{.esdhc_base = USDHC4_BASE_ADDR, .bus_width = 8,
			.gp_reset = GP_EMMC_RESET},
};
#endif

#ifdef CONFIG_MXC_SPI
int board_spi_cs_gpio(unsigned bus, unsigned cs)
{
	return (bus == 0 && cs == 0) ? GP_ECSPI1_NOR_CS : -1;
}
#endif

#ifdef CONFIG_CMD_FBPANEL
void board_enable_lvds(const struct display_info_t *di, int enable)
{
	if (enable) {
		mdelay(200);
		gpio_set_value(GP_BACKLIGHT_LVDS_PWM1, enable);
		mdelay(10);
		/* enable backlight PWM 3 */
		SETUP_IOMUX_PADS(lvds_led_pwm_pads);
		pwm_init(2, 0, 0);
		/* 20 KHz, duty cycle 30%, period: 0.05 ms */
		pwm_config(2, 15000, 50000);
		pwm_enable(2);
		mdelay(10);
		gpio_set_value(GP_LVDS_LED_EN, enable);
	} else {
		gpio_set_value(GP_LVDS_LED_EN, enable);
		SETUP_IOMUX_PADS(lvds_led_pwm_gpio_pads);
		mdelay(10);
		gpio_set_value(GP_BACKLIGHT_LVDS_PWM1, enable);
	}
}

static const struct display_info_t displays[] = {
	/* EDID at 0x50 */
	VD_M101NWWB(LVDS, fbp_detect_i2c, fbp_bus_gp(0, 0, GP_LVDS_BKL_EN, 0), 0x50),
	/* ft5x06 */
	VD_HANNSTAR7(LVDS, fbp_detect_i2c, fbp_bus_gp(2, 0, GP_LVDS_BKL_EN, 0), 0x38, FBTS_FT5X06),
	VD_AUO_B101EW05(LVDS, NULL, fbp_bus_gp(2, 0, GP_LVDS_BKL_EN, 0), 0x38, FBTS_FT5X06),
	VD_LG1280_800(LVDS, NULL, fbp_bus_gp(2, 0, GP_LVDS_BKL_EN, 0), 0x38, FBTS_FT5X06),
	VD_DT070BTFT(LVDS, NULL, fbp_bus_gp(2, 0, GP_LVDS_BKL_EN, 0), 0x38, FBTS_FT5X06),
	VD_WSVGA(LVDS, NULL, fbp_bus_gp(2, 0, GP_LVDS_BKL_EN, 0), 0x38, FBTS_FT5X06),
	VD_TM070JDHG30(LVDS, NULL, fbp_bus_gp(2, 0, GP_LVDS_BKL_EN, 0), 0x38, FBTS_FT5X06),
	VD_ND1024_600(LVDS, fbp_detect_i2c, fbp_bus_gp(2, 0, GP_LVDS_BKL_EN, 0), 0x38, FBTS_FT5X06),

	/* ili210x */
	VD_AMP1024_600(LVDS, fbp_detect_i2c, fbp_bus_gp(2, 0, GP_LVDS_BKL_EN, 0), 0x41, FBTS_ILI210X),

	/* egalax_ts */
	VD_HANNSTAR(LVDS, fbp_detect_i2c, fbp_bus_gp(2, 0, GP_LVDS_BKL_EN, 0), 0x04, FBTS_EGALAX),
	VD_LG9_7(LVDS, NULL, 2, 0x04, FBTS_EGALAX),

	VD_SHARP_LQ101K1LY04(LVDS, NULL, fbp_bus_gp(0, 0, GP_LVDS_BKL_EN, 0), 0x00),
	VD_WXGA(LVDS, NULL, fbp_bus_gp(0, 0, GP_LVDS_BKL_EN, 0), 0x00),
	VD_LD070WSVGA(LVDS, NULL, 0, 0x00),
	VD_WVGA(LVDS, NULL, fbp_bus_gp(0, 0, GP_LVDS_BKL_EN, 0), 0x00),
	VD_AA065VE11(LVDS, NULL, 0, 0x00),
	VD_VGA(LVDS, NULL, fbp_bus_gp(0, 0, GP_LVDS_BKL_EN, 0), 0x00),
};
#define display_cnt	ARRAY_SIZE(displays)
#else
#define displays	NULL
#define display_cnt	0
#endif


static const unsigned short gpios_out_low[] = {
	GP_LVDS_BKL_EN,
	GP_BACKLIGHT_LVDS_PWM1,
	GP_LVDS_LED_EN,
	GP_LVDS_BIST,
	GP_BT_RFKILL_RESET,
	GP_RGMII_PHY_RESET,
	GP_GPS_RESET,
	GP_MODEM_RESET,
	GP_MODEM_PULSE_ON,
	GP_LED_VIBRATOR,
	GP_I2C1MUX_A0,
	GP_I2C1MUX_A1,
	GP_I2C1MUX_RESET,
	GP_FP_RESET_N,
	GP_FP_LE_EN,
	GP_OV5640_RESET,
	GP_OV5640_MIPI_RESET,
	GP_GT928_RESET,
	GP_FLASH_STROBE,
	GP_FLASH_HW_EN,
	GP_TORCH_EN,
	GP_FLASH_TX,
	GP_LVDS_LED_PWM,
	GP_PWM2,
	GP_REG_2P8V_EN,
	GP_REG_FP_3P3V_EN,
	GP_REG_FP_5V_EN,
	GP_REG_USBOTG,
	GP_REG_VBUS4,
	GP_REG_WLAN_EN,
	GP_USBH1_HUB_RESET,
	GP_EMMC_RESET,
};

static const unsigned short gpios_out_high[] = {
	GP_ECSPI1_NOR_CS,
	GP_ECSPI2_CS,
	GP_STAT_LED2,
	GP_STAT_LED1,
	GP_STAT_LED3,
	GP_OV5640_POWER_DOWN,
	GP_OV5640_STROBE,
	GP_OV5640_MIPI_POWER_DOWN,
	GP_3P7_EN,
	GP_3P7_BYPASS_EN,
};

static const unsigned short gpios_in[] = {
	GPIRQ_ENET_PHY,
	GP_GPIOKEY_POWER,
	GP_GPIOKEY_VOL_DN,
	GP_GPIOKEY_VOL_UP,
	GP_GPS_HEARTBEAT,
	GP_GPS_INT,
	GP_GPIO_A,
	GP_GPIO_B,
	GP_SLEEP_STAT,
	GP_SIM_DETECT,
	GP_SIM_IO,
	GP_SIM_CLK,
	GP_BT_CLK_REQ,
	GP_BT_HOST_WAKE,
	GP_WIFI_QOW,
	GP_TP68,
	GP_TP71,
	GP_TP72,
	GP_TP73,
	GP_TP74,
	GP_TP78,
	GP_TP85,
	GP_TP86,
	GP_TP90,
	GP_TP91,
	GP_TP92,
	GP_TP97,
	GP_TP104,
	GP_TP131,
	GP_TP_R368,
	GP_WM8960_MIC_DET,
	GP_WM8960_HP_DET,
	GPIRQ_MPU9250_INT,
	GP_FP_SENSOR_STAT,
	GPIRQ_RTC,
	GPIRQ_TAMPER,
	GPIRQ_TAMPER2,
	GPIRQ_LIGHTSENSOR,
	GPIRQ_GT928,
	GPIRQ_MAX77818_INOK,
	GPIRQ_MAX77818_WCINOK,
	GPIRQ_MAX77818,
	GPIRQ_WIFI,
	GP_WIFI_WAKE,
	GP_USDHC3_CD,
};

int board_early_init_f(void)
{
	set_gpios_in(gpios_in, ARRAY_SIZE(gpios_in));
	set_gpios(gpios_out_low, ARRAY_SIZE(gpios_out_low), 0);
	set_gpios(gpios_out_high, ARRAY_SIZE(gpios_out_high), 1);
	SETUP_IOMUX_PADS(init_pads);
	return 0;
}

void board_poweroff(void)
{
	struct snvs_regs *snvs = (struct snvs_regs *)(SNVS_BASE_ADDR);

	max77823_boost_power(0);
	writel(0x60, &snvs->lpcr);
	mdelay(500);
}

int board_init(void)
{
	common_board_init(i2c_pads, I2C_BUS_CNT, IOMUXC_GPR1_OTG_ID_GPIO1,
			displays, display_cnt, 0);
	return 0;
}

const struct button_key board_buttons[] = {
	{"reset",	GP_GPIOKEY_POWER,	'r', 1},
	{"vol_dn",	GP_GPIOKEY_VOL_DN,	'-', 1},
	{"vol_up",	GP_GPIOKEY_VOL_UP,	'+', 1},
	{"tamper",	TAMPER_CHECK,		't', 1},
	{"tp74",	GP_TP74,		'4', 0},
	{NULL, 0, 0, 0},
};

#ifdef CONFIG_CMD_BMODE
const struct boot_mode board_boot_modes[] = {
	/* 4 bit bus width */
	{"mmc0",	MAKE_CFGVAL(0x40, 0x30, 0x00, 0x00)},
	{"mmc1",	MAKE_CFGVAL(0x60, 0x58, 0x00, 0x00)},	/* 8-bit eMMC */
	{NULL,		0},
};
#endif

static int _do_poweroff(cmd_tbl_t *cmdtp, int flag, int argc, char * const argv[])
{
	board_poweroff();
	return 0;
}

U_BOOT_CMD(
	poweroff, 70, 0, _do_poweroff,
	"power down board",
	""
);
